# Copyright (C) 2008-2012 eBox Technologies S.L.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License, version 2, as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

package EBox::Printers;

use strict;
use warnings;

use base qw(EBox::Module::Service EBox::FirewallObserver
            EBox::Model::ModelProvider EBox::LogObserver
            EBox::Model::CompositeProvider);

use EBox::Gettext;
use EBox::Config;
use EBox::Service;
use EBox::Menu::Item;
use EBox::Sudo;
use EBox::PrinterFirewall;
use EBox::Printers::LogHelper;
use Net::CUPS::Destination;
use Net::CUPS;
use Error qw(:try);

use constant CUPSD => '/etc/cups/cupsd.conf';

sub _create
{
    my $class = shift;
    my $self = $class->SUPER::_create(name => 'printers',
                                      printableName => __('Printer Sharing'),
                                      @_);
    bless($self, $class);
    $self->{'cups'} = new Net::CUPS;
    return $self;
}

# Method: actions
#
#   Override EBox::Module::Service::actions
#
sub actions
{
    return [
    {
        'action' => __('Create spool directory for printers'),
        'reason' => __('Zentyal will create a spool directory ' .
                       'under /var/spool/samba'),
        'module' => 'printers'
    },
    {
        'action' => __('Create log table'),
        'reason' => __('Zentyal will create a new table into its log database ' .
                       'to store printers logs'),
        'module' => 'printers'
    },
    {
        'action' => __x('Disable {server} init script', server => 'cups'),
        'reason' => __('Zentyal will take care of start and stop ' .
                       'the service'),
        'module' => 'printers',
    }
    ];
}

# Method: usedFiles
#
#   Override EBox::Module::Service::files
#
sub usedFiles
{
    return [
    {
        'file' => CUPSD,
        'reason' => __('To configure cupsd'),
        'module' => 'printers',
    },
    ];
}

# Method: initialSetup
#
# Overrides:
#   EBox::Module::Base::initialSetup
#
sub initialSetup
{
    my ($self, $version) = @_;

    # Execute initial-setup script
    $self->SUPER::initialSetup($version);

    # Add IPP service only if installing the first time
    unless ($version) {
        my $firewall = EBox::Global->modInstance('firewall');
        $firewall->addInternalService(
                        'name' => 'ipp',
                            'description' => __d('Cups printer server port'),
                            'protocol' => 'tcp',
                            'sourcePort' => 'any',
                            'destinationPort' => 631,
                           );
        $firewall->saveConfigRecursive();
    }
}

# Method: enableActions
#
#   Override EBox::Module::Service::enableActions
#
sub enableActions
{
    my ($self) = @_;

    # Execute enable-module script
    $self->SUPER::enableActions();

    # Write conf file for the first time using the template,
    # next times only some lines will be overwritten to
    # avoid conflicts with CUPS interface
    $self->writeConfFile(CUPSD, 'printers/cupsd.conf.mas',
                         [ addresses => $self->_ifaceAddresses() ]);
}

# Method: enableService
#
# Overrides:
#
#  <EBox::Module::Service::enableService>
#
sub enableService
{
    my ($self, $status) = @_;

    $self->SUPER::enableService($status);

    my $samba = EBox::Global->modInstance('samba');
    $samba->setPrinterService($status);
}

sub restoreDependencies
{
    return [ 'network' ];
}

# Method: modelClasses
#
# Overrides:
#
#       <EBox::Model::ModelProvider::modelClasses>
#
sub modelClasses
{
    my ($self) = @_;

    return [ 'EBox::Printers::Model::CUPS' ];
}

# Method: compositeClasses
#
# Overrides:
#
#       <EBox::Model::ModelProvider::compositeClasses>
#
sub compositeClasses
{
    my ($self) = @_;

    return [ 'EBox::Printers::Composite::General' ];
}

sub firewallHelper
{
    my ($self) = @_;

    if ($self->isEnabled()) {
        return new EBox::PrinterFirewall();
    }
    return undef;
}

sub _preSetConf
{
    my ($self) = @_;

    try {
        # Stop CUPS in order to force it to dump the conf to disk
        $self->_stopService();
    } otherwise {};
}

# Method: _setConf
#
#   Override EBox::Module::Base::_setConf
#
sub _setConf
{
    my ($self) = @_;

    $self->_mangleConfFile(CUPSD, addresses => $self->_ifaceAddresses());
}

sub _ifaceAddresses
{
    my ($self) = @_;

    my $net = EBox::Global->modInstance('network');
    my $ifacesModel = $self->model('CUPS');
    my @addresses;
    foreach my $row (@{$ifacesModel->enabledRows()}) {
        my $iface = $ifacesModel->row($row)->valueByName('iface');
        my $address = $net->ifaceAddress($iface);
        next unless $address;
        push (@addresses, $address);
    }

    return \@addresses;
}

sub _mangleConfFile
{
    my ($self, $path, %params) = @_;

    my $newContents = '';
    my @oldContents = File::Slurp::read_file($path);

    foreach my $line (@oldContents) {
        if ($line =~ m{^\s*Listen\s}) {
            # listen statement, removing
            next;
        }  elsif ($line =~ m{^\s*SSLListen\s}) {
            # ssllisten statement, removing
            next;
        } elsif ($line =~m/ by Zentyal,/) {
            # zentyal added skipping
            next;
        }

        $newContents .= $line;
    }

    $newContents .= <<END;
# Added by Zentyal, don't modify or add more Liste/SSLListen statements
Listen localhost:631
Listen /var/run/cups/cups.sock
END
    foreach my $address (@{ $params{addresses} }) {
        $newContents .= "SSLListen $address:631\n";
    }

    EBox::Module::Base::writeFile($path, $newContents);
}


sub _daemons
{
    return [
        {
            'name' => 'cups',
            'type' => 'init.d',
            'pidfiles' => ['/var/run/cups/cupsd.pid'],
        }
    ];
}

sub menu
{
    my ($self, $root) = @_;

    my $item = new EBox::Menu::Item('name' => 'Printers Sharing',
                                    'url' => 'Printers/Composite/General',
                                    'text' => $self->printableName(),
                                    'separator' => 'Office',
                                    'order' => 550);

    $root->add($item);
}

# Method: dumpConfig
#
#   Overrides EBox::Module::Base::dumpConfig
#
sub dumpConfig
{
    my ($self, $dir, %options) = @_;

    $self->_stopService();

    my @files = ('/etc/cups/printers.conf', '/etc/cups/ppd');
    my $backupFiles = '';
    foreach my $file (@files) {
        if (EBox::Sudo::fileTest('-e', $file)) {
            $backupFiles .= " $file";
        }
    }
    EBox::Sudo::root("tar cf $dir/etc_cups.tar $backupFiles");

    $self->_startService();
}

# Method: restoreConfig
#
#   Overrides EBox::Module::Base::dumpConfig
#
sub restoreConfig
{
    my ($self, $dir) = @_;

    if (EBox::Sudo::fileTest('-f', "$dir/etc_cups.tar")) {
        try {
            $self->_stopService();
            EBox::Sudo::root("tar xf $dir/etc_cups.tar -C /");
            $self->_startService();
        } otherwise {
            EBox::error("Error restoring cups config from backup");
        };
    } else {
        # This case can happen with old backups
        EBox::warn('Backup doesn\'t contain CUPS configuration files');
    }
}

# Method: networkPrinters
#
#   Returns the printers configured as network printer
#
# Returns:
#
#   array ref - holding the printer id's
#
sub networkPrinters
{
    my ($self) = @_;

    my @ids;
# FIXME: This should be get using Net::CUPS as we are not storing
# printers in our config anymore
#    foreach my $printer (@{$self->printers()}) {
#        my $conf = $self->methodConf($printer->{id});
#        push (@ids, $printer->{id}) if ($conf->{method} eq 'network');
#    }

    return \@ids;
}


# Impelment LogHelper interface

sub tableInfo
{
    my ($self) = @_;

    my $titles = { 'job' => __('Job ID'),
                   'printer' => __('Printer'),
                   'username' => __('User'),
                   'timestamp' => __('Date'),
                   'event' => __('Event')
                 };
    my @order = ('timestamp', 'job', 'printer', 'username', 'event');
    my $events = {
                      'queued' => __('Queued'),
                      'completed' => __('Completed'),
                      'canceled' => __('Canceled'),
                     };

    return [{
             'name' => __('Printers'),
             'tablename' => 'printers_jobs',
             'titles' => $titles,
             'order' => \@order,
             'timecol' => 'timestamp',
             'filter' => ['printer', 'username'],
             'events' => $events,
             'eventcol' => 'event'

            }];
}

sub consolidateReportQueries
{
    return [
            {
             'target_table' => 'printers_jobs_report',
             'query' => {
                         'select' => 'printer,event,count(*) as nJobs',
                         'from' => 'printers_jobs',
                         'group' => 'printer,event',
                        },
             'quote' => { printer => 1 },
            },
            {
             'target_table' => 'printers_jobs_by_user_report',
             'query' => {
                         'select' => 'username,event,count(*) as nJobs',
                         'from' => 'printers_jobs',
                         'group' => 'username,event',
                        },
             'quote' => { username => 1 },
            },
            {
             target_table => 'printers_usage_report',
             'query' => {
                         'select' => 'printers_jobs.printer, SUM(pages) AS pages, COUNT(DISTINCT printers_jobs.username) AS users',
                          'from' => 'printers_pages,printers_jobs',
                          'group' => 'printers_jobs.printer',
                          'where' => q{(printers_jobs.job = printers_pages.job) and(event='queued')}
                        },
             'quote' => { printer => 1 },
            }
           ];
}

# Method: report
#
# Overrides:
#   <EBox::Module::Base::report>
sub report
{
    my ($self, $beg, $end, $options) = @_;

    my $report = {};

    my $db = EBox::DBEngineFactory::DBEngine();

    my @events = qw(queued canceled completed);

    my %eventsByPrinter;
    foreach my $event  (@events) {
        my $results = $self->runMonthlyQuery($beg, $end, {
           'select' => q{printer, SUM(nJobs)},
           'from' => 'printers_jobs_report',
           'group' => 'printer',
           'where' => qq{event='$event'},
                                                         },
           { 'key' => 'printer' });

        while (my ($printer, $res) = each %{ $results }) {
            if (not exists $eventsByPrinter{$printer}) {
                $eventsByPrinter{$printer} = {};
            }
            $eventsByPrinter{$printer}->{$event} = $res->{sum};
        }
    }

    $report->{eventsByPrinter} = \%eventsByPrinter;

    my %eventsByUsername;
    foreach my $event  (@events) {
        my $results = $self->runMonthlyQuery($beg, $end, {
           'select' => q{username, SUM(nJobs)},
           'from' => 'printers_jobs_by_user_report',
           'group' => 'username',
           'where' => qq{event='$event'},
                                                         },
           { 'key' => 'username' });

        while (my ($username, $res) = each %{ $results }) {
            if (not exists $eventsByUsername{$username}) {
                $eventsByUsername{$username} = {};
            }
            $eventsByUsername{$username}->{$event} = $res->{sum};
        }
    }

    $report->{eventsByUser} = \%eventsByUsername;

    my $printerUsage = $self->runMonthlyQuery($beg, $end, {
           'select' => q{printer, pages, users},
           'from' => 'printers_usage_report',
#           'group' => 'printer',
                                                         },
           { 'key' => 'printer' }
                                                    );
    # add job fields to usage report
    foreach my $printer (keys %{ $printerUsage} ) {
        if (not exists $eventsByPrinter{$printer}) {
            next;
        }

        my @jobs = @{ $eventsByPrinter{$printer}->{queued} };
        $printerUsage->{$printer}->{jobs} = \@jobs;
    }

    $report->{printerUsage} = $printerUsage;

    return $report;
}

sub logHelper
{
    my ($self) = @_;

    return (new EBox::Printers::LogHelper());
}

# Method: fetchExternalCUPSPrinters
#
#   This method returns those printers that haven been configured
#   by the user using CUPS.
#
# Returns:
#
#   Array ref - containing the printer names
#
sub fetchExternalCUPSPrinters
{
    my ($self) = @_;

    my $cups = Net::CUPS->new();

    my @printers;
    foreach my $printer ($cups->getDestinations()) {
        my $name = $printer->getName();
        push (@printers, $name);
    }
    return \@printers;
}

1;
