/*
 *  This file is part of X-File Manager XFM
 *  ----------------------------------------------------------------------
  execute.c

  (c) 2005,2006  Bernhard R. Link

  call to external programs
 *  ----------------------------------------------------------------------
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include <xfmconfig.h>

#include <assert.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <fcntl.h>
#include <X11/Intrinsic.h>
#include <stdarg.h>

#include "global.h"

#include "Fm.h"
#include "Am.h"

#include "execute.h"

int child_stdout = -1;
int child_stderr = -1;

static struct pidwaitedfor {
	struct pidwaitedfor *next;
	pid_t pid;
	external_done_proc *done;
	void *data;
} *waitedfor = NULL;

static void save_sigchld_handler(UNUSED(XtPointer closure),
		                 UNUSED(XtSignalId* id)) {
	pid_t pid;int status;
	struct pidwaitedfor **p;

	while( (pid = waitpid(-1,&status,WNOHANG)) > 0 ) {
		for( p = &waitedfor ; *p != NULL ; p = &(*p)->next ) {
			struct pidwaitedfor *w = *p;
			if( w->pid == pid ) {
				w->done(status,w->data);
				*p = w->next;
				free(w);
				return;
			}
		}
	}
}

static XtSignalId sigid;

static void sigchld_handler(UNUSED(int i)) {
	XtNoticeSignal(sigid);
}

bool execute_init(XtAppContext app) {
	struct sigaction newaction;

	sigid = XtAppAddSignal(app,save_sigchld_handler,NULL);
	newaction.sa_handler = sigchld_handler;
	newaction.sa_flags = SA_NOCLDSTOP;
	sigemptyset(&newaction.sa_mask);
	if( sigaction(SIGCHLD,&newaction,NULL) < 0 ) {
		sysError("Could not set SIGCHLD signal handler: ");
		return false;
	}
	return true;
}

static pid_t execute(const char *program,const char *directory, const char **argv) {
	pid_t pid;

	// TODO: move this into the exec?
	if( chdir(directory) != 0 ) {
		sysError("Can't chdir:");
		return -1;
	}
	if (resources.echo_actions) {
		const char **arg;
		fprintf(stderr, "[%s] %s", directory, argv[0]);
		for (arg = argv+1; *arg != NULL ; arg++)
			fprintf(stderr," '%s'",*arg);
		fprintf(stderr, "\n");
	}
	pid = fork();
	if( pid < 0 ) {
		sysError("Can't fork:");
		return -1;
	}
	if( pid == 0 ) {
		int fd,e;
		/* child: */
		fd = open("/dev/null",O_RDONLY|O_NOCTTY);
		if( fd >= 0 ) {
			(void)dup2(fd,STDIN_FILENO);
			if( fd != STDIN_FILENO )
				close(fd);
		}
		if( child_stdout >= 0 )
			dup2(child_stdout,STDOUT_FILENO);
		if( child_stderr >= 0 )
			dup2(child_stderr,STDERR_FILENO);
		if( child_stderr >= 3 && child_stdout != child_stderr )
			close(child_stderr);
		if( child_stdout >= 3 )
			close(child_stdout);
		/* hope the rest of the fds is close on exec */
		execvp(program,(char**)argv);
		e = errno;
		fprintf(stderr,"Error executing '%s' in directory '%s': %d=%s\n",argv[0],directory,e,strerror(e));
		raise(SIGUSR2);
		exit(EXIT_FAILURE);
	}
	return pid;
}

bool execute_and_wait(const char *program,const char *directory, const char **argv) {
	pid_t pid,w;
	int status;

	pid = execute(program,directory,argv);
	if( pid < 0 ) {
		return false;
	}
	w = waitpid(pid, &status, 0);
	return w > 0 && WIFEXITED(status) && WEXITSTATUS(status) == 0;
}

void execute_external_program(const char *program,const char *directory, const char **argv, external_done_proc *done, void *priv_data) {
	pid_t pid,w;
	int status;

	zzz();
	XFlush(XtDisplay(aw.shell));
	pid = execute(program,directory,argv);
	if( pid < 0 ) {
		wakeUp();
		return;
	}
	/* parent */
	sleep(1);
	w = waitpid(pid,&status,WNOHANG);
	if( w > 0 ) {
		if( WIFSIGNALED(status) && WSTOPSIG(status) == SIGUSR2 ) {
			fprintf(stderr,"Error executing '%s' in directory '%s'\n",argv[0],directory);
		}
		if( done != NULL )
			done(status,priv_data);
	} else if( done != NULL ) {
		struct pidwaitedfor *wf;
		wf = (struct pidwaitedfor*)XtMalloc(sizeof(struct pidwaitedfor));
		wf->next = waitedfor;
		wf->pid = pid;
		wf->done = done;
		wf->data = priv_data;
		waitedfor = wf;
	}
	
	wakeUp();
}

const char **appendArgs(const char **old, int number, ...) {
	const char **args,**p;
	va_list ap;
	int c;

	c = 0;
	if( old != NULL)
		for( p = old ; *p != NULL ; p++ )
			c++;

	args = (const char **)XtMalloc((c+number+1)*sizeof(const char*));
	p = args;
	while( c > 0 ) {
		*p = old[p-args];
		p++;
		c--;
	}
	va_start(ap,number);
	while( number > 0 ) {
		const char *argument = va_arg(ap,const char*);
		assert(argument != NULL);
		*(p++) = argument;
		number--;
	}
	assert( va_arg(ap,const char*) == NULL );
	va_end(ap);
	*(p++) = NULL;
	return args;
}

const char **makeShellArgs(const char *action, const char **arguments) {
	const char **args,**p;
	int c,ac;

	c = 0;
	for( p = resources.shell ; *p != NULL ; p++ )
		c++;
	if( arguments != NULL ) {
		ac = 1;
		for( p = arguments ; *p != NULL ; p++ )
			ac++;
	} else
		ac = 0;

	args = (const char **)XtMalloc((c+ac+3)*sizeof(const char*));
	p = args;
	while( c > 0 ) {
		*p = resources.shell[p-args];
		p++;
		c--;
	}
	*(p++) = action;
	if( ac > 0 ) {
		*(p++) = "sh";
		ac--;
		while( ac-- > 0 ) {
			*(p++) = *(arguments++);
		}
	}
	*(p++) = NULL;
	return args;
}
