/*
 *  This file is part of X-File Manager XFM
 *  ----------------------------------------------------------------------
  FmOps.c

  (c) Simon Marlow 1990-92
  (c) Albert Graef 1994

  modified 2004,2005,2007 by Bernhard R. Link (see Changelog)

  Various file manipulation operations and other useful routines
 *  ----------------------------------------------------------------------
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <xfmconfig.h>

#include <assert.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <X11/Intrinsic.h>

#include "Fm.h"

/* split a string into substrings delimited by a given character */

char *split(char *s, char c)
{
  static char *t;
  if (!s)
    s = t;
  t = s;
  if (t)
    while ((t = strchr(t, c)) && t>s && t[-1]=='\\') t++;
  if (t)
    *t++ = '\0';
  return s;
}

/*-------------------------------------------------------------------------*/

/* expand escapes in a string */

char *expand(char *s, char *t, char *c)
{
  char *s0 = s;
  for (; *t; t++) {
    if (strchr(c, *t))
      *s++ = '\\';
    *s++ = *t;
  }
  *s = '\0';
  return s0;
}

void fprintexpand(FILE *f, const char *t, const char *c)
{
  for (; *t; t++) {
    if (strchr(c, *t))
      fputc('\\',f);
    fputc(*t,f);
  }
}

/*-------------------------------------------------------------------------*/

/* remove escapes from a string */

char *strparse(char *s, const char *t, const char *c)
{
  char *s0 = s;
  for (; *t; t++)
    if (*t != '\\')
      *s++ = *t;
    else if (*++t) {
      if (!strchr(c, *t))
	*s++ = '\\';
      *s++ = *t;
    } else
      break;
  *s = '\0';
  return s0;
}
      
/*-------------------------------------------------------------------------*/

/* expand filename */

char *dyn_fnexpand(const char *fn) {
	char *n;
  	size_t l_fn,l_home;

	if( fn == NULL )
		return NULL;
	if( fn[0] != '~' || (fn[1] != '\0' && fn[1] != '/') )
		return xstrdup(fn);
	l_home = strlen(user.home);
	l_fn   = strlen(fn);
  	if( l_home >= SIZE_MAX - l_fn ) {
		abort();
	}
#ifdef MAXPATHLEN
  	if( l_fn+l_home-1 >= MAXPATHLEN ) {
		fprintf(stderr,"Error: '%s' cannot be expanded within MAXPATHLEN=%ld\n",fn,(long)MAXPATHLEN);
		exit(EXIT_FAILURE);
	}
#endif
	n = XtMalloc(l_fn+l_home+1);
  	memcpy(n, user.home, l_home);
	memcpy(n+l_home, fn+1, l_fn /*-1+1*/ );
	return n;
}

char *expanddirectory(const char *dir) {
	if( *dir )
		return dyn_fnexpand(dir);
	else
		return xstrdup(user.home);
}

char *home_and_env_expand(const char *fn) {
	char *n,*r;
  	size_t l,l_home;
	const char *envs[20];
	size_t envlen[20];
	size_t envskip[20];
	int envnum = 0;
	int i;
	const char *p;
	char pid[5];

	if( fn == NULL )
		return NULL;
	if( fn[0] != '~' || (fn[1] != '\0' && fn[1] != '/') )
		l_home = 0;
	else {
		l_home = strlen(user.home);
		fn++;		
	}
	l = l_home;
	p = fn;
	while( *p != '\0' ) {
		const char *envstart,*envend;
		if( p[0] != '$' || p[1] != '{' || envnum >= 20 ) {
			if( l == SIZE_MAX )
				abortXfm("overlong string\n");
			l++;
			p++;
			continue;
		}
		envstart = p+2;
		envend = strchr(envstart,'}');
		if( envend == NULL ) {
			if( l == SIZE_MAX )
				abortXfm("overlong string\n");
			if( envnum < 20 ) {
				envlen[envnum] = 0;
				envskip[envnum] = 0;
				envnum++;
			}
			l++;
			p++;
			continue;
		}
		if( envend-envstart == 3 && envstart[0] == 'P' 
				&& envstart[1] == 'I' 
				&& envstart[2] == 'D' ) {
			pid_t pidn = getpid();
			pid[0] = 'a'+(pidn & 0xF);
			pid[1] = 'a'+((pidn>>4) & 0xF);
			pid[2] = 'a'+((pidn>>8) & 0xF);
			pid[3] = 'a'+((pidn>>12) & 0xF);
			pid[4] = '\0';
			envs[envnum] = pid;
		} else {
			char *envname;
			envname = XtMalloc(envend-envstart+1);
			memcpy(envname,envstart,envend-envstart);
			envname[envend-envstart] = '\0';
			envs[envnum] = getenv(envname);
			free(envname);
		}
		if( envs[envnum] == NULL )
			envlen[envnum] = 0;
		else {
			size_t le = strlen(envs[envnum]);
			if( l >= SIZE_MAX-le )
				abortXfm("overlong string\n");
			l += le;
			envlen[envnum] = le;
		}
		envend++;
		envskip[envnum] = envend-p;
		envnum++;
		p = envend;
	}
	r = XtMalloc(l+1);
	n = r;
	if( l_home > 0 ) {
  		memcpy(n, user.home, l_home);
		n += l_home;
	}
	p = fn; i = 0;
	while( *p != '\0' ) {
		if( p[0] != '$' || p[1] != '{' || i >= 20 ) {
			*(n++) = *(p++);
		} else {
			if( envlen[i] > 0 ) {
				memcpy(n,envs[i],envlen[i]);
				n += envlen[i];
			}
			p += envskip[i];
			i++;
		}
	}
	*n = '\0';
	assert( (size_t)(n-r) == l );
	return r;
}

/*---------------------------------------------------------------------------*/

/* match a pattern with a filename, returning nonzero if the match was
   correct */

/* Currently only *, ? and [...] (character classes) are recognized, no curly
   braces. An escape mechanism for metacharacters is also missing. This could
   be implemented more efficiently, but the present simple backtracking
   routine does reasonably well for the usual kinds of patterns. -ag */

int fnmatch(char *pattern, char *fn)
{
  char *start;
  
  for (;; fn++, pattern++) {
    
    switch (*pattern) {
      
    case '?':
      if (!*fn)
	return 0;
      break;
      
    case '*':
      pattern++;
      do
	if (fnmatch(pattern,fn))
	  return 1;
      while (*fn++);
      return 0;
      
    case '[':
      start = pattern+1;
      do {
      next:
	pattern++;
	if (*pattern == ']')
	  return 0;
	else if (pattern[0] == '-' && pattern > start && pattern[1] != ']') {
	  if (pattern[-1] <= *fn && *fn <= pattern[1])
	    break;
	  else {
	    start = (++pattern)+1;
	    goto next;
	  }
	}
      } while (*fn != *pattern);
      while (*pattern != ']')
	if (!*pattern++)
	  return 0;
      break;
      
    default:
      if (*fn != *pattern)
	return 0;
    }
    
    if (!*fn)
      return 1;
  };
}

/*-------------------------------------------------------------------------*/

/* check whether one path is the prefix of another */

int prefix(const char *s, const char *t)
{
  int l = strlen(s);

  return !strncmp(s, t, l) && (s[l-1] == '/' || t[l] == '\0' || t[l] == '/');
}

/*-------------------------------------------------------------------------*/

/* check whether a file exists */

int exists(const char *path)
{
  struct stat stats;

  return (!lstat(path, &stats));
}

/*-------------------------------------------------------------------------*/

/* find file on given search path */

char *searchPath(char *s1, char *p, const char *s2)
{
  char           *s, *t;
  size_t s2len = strlen(s2);

  /* TODO: rewrite this or at least check for maximum lengths */
  if (*s2 == '/' || !p)
	  return memcpy(s1, s2, s2len+1);
  for (s = p; *s; s = t) {
    size_t l;
    if (!(t = strchr(s, ':')))
      t = strchr(s, 0);
    if (s == t) goto next;
    if (s[0] == '.') {
      if ((t = s+1))
	s = t;
      else if (s[1] == '/')
	s += 2;
    }
    l = t-s;
    memcpy(s1, s, l);
    if (l > 0 && s1[l - 1] != '/')
      s1[l] = '/', l++;
    memcpy(s1+l, s2, s2len+1);
    if (exists(s1))
      return s1;
  next:
    if (*t) t++;
  }
  return (strcpy(s1, s2));
}

char *searchPathWithSuffix(char *s1, char *p, const char *s2, const char *suffix)
{
  char           *s, *t;
  size_t s2len = strlen(s2);
  size_t sufflen = suffix?strlen(suffix):0;

  /* TODO: rewrite this or at least check for maximum lengths */
  
  if (*s2 == '/' || !p) {
	  if ( suffix == NULL )
		  return memcpy(s1, s2, s2len+1);
	  else {
		  memcpy(s1, s2, s2len);
		  memcpy(s1+s2len, suffix, sufflen+1);
		  return s1;
	  }
  }
  for (s = p; *s; s = t) {
    size_t l;
    if (!(t = strchr(s, ':')))
      t = strchr(s, 0);
    if (s == t) goto next;
    if (s[0] == '.') {
      if ((t = s+1))
	s = t;
      else if (s[1] == '/')
	s += 2;
    }
    l = t-s;
    memcpy(s1, s, l);
    if (l > 0 && s1[l - 1] != '/')
      s1[l] = '/', l++;
    memcpy(s1+l, s2, s2len+1);
    if (sufflen > 0 )
      memcpy(s1+l+s2len,suffix,sufflen+1);
    if (exists(s1))
      return s1;
  next:
    if (*t) t++;
  }
  return NULL;
}

/* The following operations return zero on success and -1 on error, with errno
   set appropriately */

/*-------------------------------------------------------------------------*/

/* create a new file */

int create(const char *path, mode_t mode)
{
  int file = open(path, O_WRONLY|O_CREAT|O_EXCL, mode);

  if (file == -1 || close(file))
    return -1;
  else
    return 0;
}

/*-------------------------------------------------------------------------*/

/* recursive copy operation */

static ino_t *inodes;
static int copyfile(const char *oldpath, const char *newpath);
static int copydir(int n_inodes, struct stat *oldstats,
		   const char *oldpath, const char *newpath);
static int copy(int n_inodes, const char *oldpath, const char *newpath);

int rcopy(const char *oldpath, const char *newpath)
{
  inodes = NULL;
  return copy(0, oldpath, newpath);
}

static int copyfile(const char *oldpath, const char *newpath)
{
  struct stat stats;
  int src = -1, dest = -1, n, errno_ret;
  char buf[BUFSIZ];

  if ((src = open(oldpath, O_RDONLY)) == -1 || stat(oldpath, &stats))
    goto err;
  else if ((dest = creat(newpath, stats.st_mode)) == -1)
    goto err;

  while ( (n = read(src, buf, BUFSIZ)) != 0)
    if ( n == -1 || write(dest, buf, n) != n )
      goto err;

  if (close(src)) {
    src = -1;
    goto err;
  } else
    return close(dest);

 err:

  errno_ret = errno;
  if (src != -1) close(src);
  if (dest != -1) close(dest);
  errno = errno_ret;
  return -1;
}

static int copydir(int n_inodes, struct stat *oldstats,
		   const char *oldpath, const char *newpath)
{
  DIR *dir;
  struct dirent *entry;
  int i, ol = strlen(oldpath), nl = strlen(newpath);
  struct stat newstats;

  for (i = n_inodes-1; i >= 0; i--)
    if (inodes[i] == oldstats->st_ino) {
      errno = EINVAL;
      return -1;
    }

  if ((mkdir(newpath, user.umask & 0777) < 0 && errno != EEXIST) ||
       lstat(newpath, &newstats) ||
       !(dir = opendir(oldpath)))
    return -1;

  inodes = (ino_t *) XTREALLOC(inodes, (n_inodes+1)*sizeof(ino_t));
  inodes[n_inodes++] = newstats.st_ino;

  for(i = 0; (entry = readdir(dir)); i++)
    if (entry->d_name[0] != '.' || (entry->d_name[1] != '\0'
				    && (entry->d_name[1] != '.' ||
					entry->d_name[2] != '\0'))) {
      int ol1 = ol, nl1 = nl, l = strlen(entry->d_name);
      char *oldpath1 = alloca(ol1+l+2);
      char *newpath1 = alloca(nl1+l+2);

      strcpy(oldpath1, oldpath);
      strcpy(newpath1, newpath);
      if (oldpath1[ol1-1] != '/')
	oldpath1[ol1++] = '/';
      if (newpath1[nl1-1] != '/')
	newpath1[nl1++] = '/';
      strcpy(oldpath1+ol1, entry->d_name);
      strcpy(newpath1+nl1, entry->d_name);
      if (copy(n_inodes, oldpath1, newpath1)) {
	/* take care of recursive errors */
	sysErrorFmt("Error copying %s:", oldpath1);
      }
    }

  /* bug fixed by Walt Killjoy (ngogn@clark.net) */
  if (n_inodes > 1)
    inodes = (ino_t *) XTREALLOC(inodes, (n_inodes-1)*sizeof(ino_t));
  else
    XTFREE(inodes);
  return closedir(dir);
}

static int copy(int n_inodes, const char *oldpath, const char *newpath)
{
  struct stat stats, stats2;

  if (lstat(oldpath, &stats))
    return -1;
  else if (!lstat(newpath, &stats2) && !S_ISDIR(stats2.st_mode))
    unlink(newpath);

  /* Directory: copy recursively */
  if (S_ISDIR(stats.st_mode))
    return copydir(n_inodes, &stats, oldpath, newpath);

  /* Regular file: copy block by block */
  else if (S_ISREG(stats.st_mode))
    return copyfile(oldpath, newpath);

  /* Fifo: make a new one */
  else if (S_ISFIFO(stats.st_mode))
    return mkfifo(newpath, user.umask & 0666);

  /* Device: make a new one */
  else if (S_ISBLK(stats.st_mode) || S_ISCHR(stats.st_mode) ||
	     S_ISSOCK(stats.st_mode))
    return mknod(newpath, user.umask & 0666, stats.st_rdev);

  /* Symbolic link: make a new one */
  else if (S_ISLNK(stats.st_mode)) {
    char lnk[MAXPATHLEN+1];
    int l = readlink(oldpath, lnk, MAXPATHLEN);

    if (l<0)
      return -1;
    lnk[l] = '\0';
    return(symlink(lnk, newpath));
  }

  /* This shouldn't happen */
  else {
    error("Unrecognized file type:", oldpath);
    return 0;
  }
}

/*-------------------------------------------------------------------------*/

/* move files (by copy/del across file systems) */

int rmove(const char *oldpath, const char *newpath)
{
  struct stat stats;

  if (lstat(oldpath, &stats))
    return -1;
  else if (unlink(newpath) && errno != ENOENT)
    return -1;
  if (rename(oldpath, newpath))
    if (errno == EXDEV)
      if (rcopy(oldpath, newpath))
	return -1;
      else if (stat(oldpath, &stats))
	return -1;
      else if (S_ISDIR(stats.st_mode))
	return rdel(oldpath);
      else
	return unlink(oldpath);
    else
      return -1;
  else
    return 0;
}

/*-------------------------------------------------------------------------*/

/* recursive delete */

int rdel(const char *path)
{
  struct stat stats;

  if (lstat(path, &stats))
    return -1;

  if (S_ISDIR(stats.st_mode)) {
    DIR *dir;
    struct dirent *entry;
    int i, pl = strlen(path);
  
    if (!(dir = opendir(path)))
      return -1;

    for(i = 0; (entry = readdir(dir)); i++)
      if (entry->d_name[0] != '.' || (entry->d_name[1] != '\0'
				      && (entry->d_name[1] != '.' ||
					  entry->d_name[2] != '\0'))) {
	int pl1 = pl, l = strlen(entry->d_name);
	char *path1 = (char *)alloca(pl1+l+2);

	strcpy(path1, path);
	if (path1[pl1-1] != '/')
	  path1[pl1++] = '/';
	strcpy(path1+pl1, entry->d_name);
	if (rdel(path1)) {
	  /* take care of recursive errors */
	  sysErrorFmt("Error deleting %s:", path);
	}
    }

    if (closedir(dir))
      return -1;
    else
      return rmdir(path);
  } else
    return unlink(path);
}
/*-------------------------------------------------------------------------*/

/* like strdup, but with error */
char *xstrdup(const char *str) {
	char *r;

	r = strdup(str);
	if (r == NULL)
	  abortXfm("Out of Memory");
	return r;
}

/* concat Directory and filename  */
char *dirConcat(const char *directory,const char *name) {
	size_t dlen,nlen;
	char *r;

	dlen = (directory!=NULL)?strlen(directory):0;
	nlen = strlen(name);
	if( dlen == 0 )
		return xstrdup(name);
	if( dlen > 0 && directory[dlen-1] == '/' )
		dlen--;
	r = XtMalloc(dlen+nlen+2);
	if( r == NULL )
		return r;
	memcpy(r,directory,dlen);
	r[dlen] = '/';
	memcpy(r+dlen+1,name,nlen+1);
	return r;
}


char *spaceConcat(const char *str1, const char *str2) {
	size_t l1=strlen(str1), l2=strlen(str2);
	char *r = XtMalloc(l1+l2+2);
	memcpy(r,str1,l1);
	r[l1] = ' ';
	memcpy(r+l1+1,str2,l2);
	r[l1+1+l2] = '\0';
	return r;
}

char *suffixConcat(const char *str1, const char *str2) {
	size_t l1=strlen(str1), l2=strlen(str2);
	char *r = XtMalloc(l1+l2+2);
	memcpy(r,str1,l1);
	r[l1] = '.';
	memcpy(r+l1+1,str2,l2);
	r[l1+1+l2] = '\0';
	return r;
}

Boolean cmpCanonical(const char *dir1, const char *dir2) {
	char *realdir1, *realdir2;
	size_t l1, l2;
	Boolean result;

	realdir1 = canonicalize_file_name(dir1);
	l1 = strlen(realdir1);
	if( l1 > 0 && realdir1[l1-1] == '/' )
		l1--;
	realdir2 = canonicalize_file_name(dir2);
	l2 = strlen(realdir2);
	if( l2 > 0 && realdir2[l2-1] == '/' )
		l2--;
	if( l1 != l2 )
		result = False;
	else
		result = memcmp(realdir1, realdir2, l1) == 0;
	free(realdir1);
	free(realdir2);
	return result;
}
