/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef OWQUTECOMACCOUNT_H
#define OWQUTECOMACCOUNT_H

#include <model/account/SipAccount.h>
#include <model/webservices/QuteComWebService.h>

#include <thread/Timer.h>

#include <util/Event.h>

class NetworkObserver;

/**
 * Connects to the single sign-on (SSO) system of the QuteCom SIP service.
 *
 * @ingroup model
 * @author Tanguy Krotoff
 * @author Philippe Bernery
 * @author Mathieu Stute
 * @author Xavier Desjardins
 */
class QuteComAccount : public SipAccount, public QuteComWebService {

	friend class QuteComAccountXMLSerializer;

	friend class QuteComAccountParser;

public:
	QuteComAccount();

	QuteComAccount(const std::string & login, const std::string & password, bool rememberPassword);

	virtual ~QuteComAccount();

	QuteComAccount(const QuteComAccount & qutecomAccount);

	virtual QuteComAccount * clone() const;

	virtual QuteComAccount & operator = (const QuteComAccount & qutecomAccount);
	
	virtual bool operator == (const SipAccount & other) const;

	//virtual void init();

	/** Gets the QuteCom login (e-mail address). */
	const std::string & getQuteComLogin() const { return _qutecomLogin; }

	/** Gets the QuteCom Password. */
	const std::string & getQuteComPassword() const { return _qutecomPassword; }

	/** Sets the QuteCom login (e-mail address). */
	void setQuteComLogin(const std::string newlogin) { _qutecomLogin = newlogin; }

	/** Sets the QuteCom Password. */
	void setQuteComPassword(const std::string newpass) { _qutecomPassword = newpass; }
	
	virtual const std::string getUserProfileName() const {
		return _qutecomLogin;
	}
	
	/**
	 * @return password set by user
	 */
	virtual const std::string getUserPassword() const { return _qutecomPassword; }

	/**
	 * @return the full identity
	 * e.g: robobob666@voip.qutecom.fr
	 */
	virtual const std::string getFullIdentity() const {
		return _qutecomLogin;
	}

	virtual const std::string getVisibleName() const {
		return _qutecomLogin;
	}

	virtual SipAccountType getType() const {
		return SipAccount::SipAccountTypeQuteCom;
	}

	/**
	 * @return true if empty
	 */
	virtual bool isEmpty() const;

private:

	void copy(const QuteComAccount & qutecomAccount);

	virtual EnumSipLoginState::SipLoginState discoverNetwork();

	/**
	 * Discovers network for SSO request.
	 *
	 * Please contact network@openqutecom.com before any modifications.
	 */
	bool discoverForSSO();

	/**
	 * Discovers network for SIP.
	 *
	 * Please contact network@openqutecom.com before any modifications.
	 */
	bool discoverForSIP();

	void qutecomLoginEventHandler();

	void answerReceived(const std::string & answer, int requestId);

	void ssoTimeoutEventHandler();

	void ssoLastTimeoutEventHandler();

	std::string _qutecomLogin;

	std::string _qutecomPassword;

	Timer _ssoTimer;

	/** True if SSO request can be done with SSL. */
	bool _ssoWithSSL;

	/** True if SSO request is Ok. */
	bool _ssoRequestOk;

	/** True if SSO request is Ok and login/password are valid. */
	bool _qutecomLoginOk;

	/** True when _timer is finished. */
	bool _ssoTimerFinished;

	/** Number of testStun retry. */
	static unsigned short _testStunRetry;
};

#endif	//OWQUTECOMACCOUNT_H
