/*
 * Picviz - Parallel coordinates ploter
 * Copyright (C) 2008 Sebastien Tricaud <toady@gscore.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $Id$
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <picviz.h>
#include <cairo.h>

#include "../props/color.h"

cairo_t *cr;

/* To draw the text every N lines (pcv -LN) */
PcvCounter draw_rawtext[PICVIZ_MAX_AXES];

static void draw_text(double x, double y, double size, char *text)
{
	cairo_set_source_rgb(cr, 0, 0, 0);
	cairo_select_font_face (cr, "Sans", CAIRO_FONT_SLANT_NORMAL,
			CAIRO_FONT_WEIGHT_BOLD);
	cairo_set_font_size (cr, size);

	cairo_move_to (cr, x, y);
	cairo_show_text (cr, text);
	cairo_stroke(cr);
}

void draw_line(PicvizImage *image, PcvID axis_id, PicvizLine *line, PicvizAxisPlot *axisplot1, PicvizAxisPlot *axisplot2, PcvWidth x1, PcvHeight y1, PcvWidth x2, PcvHeight y2)
{
	char *color = NULL;
	double line_width;

	switch(engine.heatline_algo) {
		case 0: /* Default */
			color = picviz_properties_get(axisplot1->props, "color");
			break;
		case 1: /* Virus */
			color = picviz_properties_get(line->props, "color");
			break;
	}

	if (!color) color = picviz_properties_get(line->props, "color");

	cairo_set_source_rgb(cr,
			picviz_color_extract_r(color),
			picviz_color_extract_g(color),
			picviz_color_extract_b(color));

	line_width = atof(picviz_properties_get(line->props, "penwidth")) * 0.4;
	if (line_width) {
		cairo_set_line_width (cr, atof(picviz_properties_get(line->props, "penwidth")));
	}
	cairo_move_to(cr, (double)x1, (double) y1);
	cairo_line_to(cr, (double)x2, (double) y2);
	cairo_stroke(cr);

	if (engine.display_raw_data) {
		if (engine.draw_text == draw_rawtext[axis_id]) {
			PicvizAxis *axis = picviz_axis_get(image, axis_id);
			PicvizAxis *axis0 = picviz_axis_get(image, 0);
			char *print0 = picviz_properties_get(axis0->props, "print");
			char *print = picviz_properties_get(axis->props, "print");
			if (!print) print = "";
			if (!print0) print0 = "";

			if (axis_id == 1) {
				if (strcmp(print0,"false")) {
					draw_text(x1, y1, image->height / engine.font_factor, axisplot1->strval);
				}
				if (strcmp(print,"false")) {
					draw_text(x2, y2, image->height / engine.font_factor, axisplot2->strval);
				}
			} else {
				if (strcmp(print,"false")) {
					draw_text(x2, y2, image->height / engine.font_factor, axisplot2->strval);
				}
			}
			draw_rawtext[axis_id] = 1;
		} else {
			draw_rawtext[axis_id]++;
		}
	}

}

cairo_status_t outstdout(void *closure _U_, unsigned char *data, unsigned int length)
{
	if ( fwrite(data, 1, length, stdout) != length )
		return CAIRO_STATUS_WRITE_ERROR;

	return CAIRO_STATUS_SUCCESS;
}

void init(void)
{
	int i;

	for (i = 0; i <= PICVIZ_MAX_AXES; i++) {
		draw_rawtext[i] = 1; /* Defaults = draw text every line */
	}
}

void output(PicvizImage *image, char *arg _U_)
{
	PicvizAxis *axis;
	PicvizLine *line;
	cairo_surface_t *surface;

	init();

	surface = cairo_image_surface_create(CAIRO_FORMAT_RGB24,
			(int)image->width, (int)image->height);

	cr = cairo_create (surface);

	cairo_rectangle (cr, 0, 0, (int)image->width, (int)image->height);
	cairo_set_source_rgb (cr,
				picviz_color_extract_r(image->bgcolor),
				picviz_color_extract_g(image->bgcolor),
				picviz_color_extract_b(image->bgcolor));
	cairo_fill (cr);

	cairo_set_source_rgb(cr, 0, 0, 0);
	cairo_set_line_width (cr, 0.5);

	/* Draw each axis and header titles */
	llist_for_each_entry(axis, &image->axes, list) {
		/* Draw vertical lines */
		cairo_move_to(cr, (double) axis->xpos, (double)image->header_height);
		cairo_line_to(cr, (double) axis->xpos, (double)image->height);
	}

	cairo_stroke(cr);

	if (image->header_height) {
		//cairo_set_source_rgb(cr, 0, 0, 0);
		cairo_move_to(cr, (double)0, (double)image->header_height);
		cairo_line_to(cr, (double)image->width, (double)image->header_height);
		llist_for_each_entry(axis, &image->axes, list) {
			cairo_text_extents_t extents;
			char *label = picviz_properties_get(axis->props, "label");
			cairo_text_extents(cr, label, &extents);
			draw_text(axis->xpos - (extents.width/2), image->header_height - 5, image->height / engine.font_factor, label);
		}
		cairo_stroke(cr);
	}

	llist_for_each_entry(line, &image->lines, list) {
		if ( ! line->hidden ) {
			picviz_line_draw(image, line, draw_line);
		}
	}

	if (! engine.output_file ) {
		cairo_surface_write_to_png_stream(surface, (cairo_write_func_t)outstdout, NULL);
	} else {
		cairo_surface_write_to_png(surface, engine.output_file);
	}

	cairo_destroy (cr);
	cairo_surface_destroy (surface);

}


