/*
 * Copyright (c) 2016, 2023, Oracle and/or its affiliates. All rights reserved.
 * Copyright (c) 2016, 2023 SAP SE. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 *
 */

#include "precompiled.hpp"
#include "asm/macroAssembler.inline.hpp"
#include "runtime/deoptimization.hpp"
#include "runtime/frame.inline.hpp"
#include "runtime/javaThread.hpp"
#include "runtime/stubRoutines.hpp"

// Implementation of the platform-specific part of StubRoutines - for
// a description of how to extend it, see the stubRoutines.hpp file.

address StubRoutines::zarch::_partial_subtype_check = nullptr;

// Comapct string intrinsics: Translate table for string inflate intrinsic. Used by trot instruction.
address StubRoutines::zarch::_trot_table_addr = nullptr;

int StubRoutines::zarch::_atomic_memory_operation_lock = StubRoutines::zarch::unlocked;

#define __ masm->

void StubRoutines::zarch::generate_load_absolute_address(MacroAssembler* masm, Register table, address table_addr, uint64_t table_contents) {
  __ load_absolute_address(table, table_addr);

#ifdef ASSERT
  assert(table_addr != nullptr, "CRC lookup table address must be initialized by now");
  assert(*((uint32_t*)(table_addr+4)) == (uint32_t)table_contents, "Bad CRC lookup table: 0x%8.8x, expected 0x%8.8x", *((uint32_t*)(table_addr+4)), (uint32_t)table_contents);
  {
    Label L;
    __ load_const_optimized(Z_R0, table_addr);
    __ z_cgr(table, Z_R0);  // safety net
    __ z_bre(L);
    __ stop("crc_table: external word relocation required for load_absolute_address", 0x33);
    __ bind(L);
  }
  {
    Label L;
    __ load_const_optimized(Z_R0, table_contents); // crc_table: data @ offset(4)
    __ z_cl(Z_R0, Address(table, 4));  // safety net
    __ z_bre(L);
    __ z_l(Z_R0, Address(table, 4));   // Load data from memory, we know the constant we compared against.
    __ stop("crc_table: address or contents seems to be messed up", 0x22);
    __ bind(L);
  }
#endif
}

void StubRoutines::zarch::generate_load_crc_table_addr(MacroAssembler* masm, Register table) {
  const uint64_t table_contents = 0x77073096UL;  // required contents of table[1]
  generate_load_absolute_address(masm, table, StubRoutines::_crc_table_adr, table_contents);
}

void StubRoutines::zarch::generate_load_crc32c_table_addr(MacroAssembler* masm, Register table) {
  const uint64_t table_contents = 0xf26b8303UL;  // required contents of table[1]
  generate_load_absolute_address(masm, table, StubRoutines::_crc32c_table_addr, table_contents);
}


// Comapct string intrinsics: Translate table for string inflate intrinsic. Used by trot instruction.
void StubRoutines::zarch::generate_load_trot_table_addr(MacroAssembler* masm, Register table) {

  RelocationHolder   rspec = external_word_Relocation::spec((address)_trot_table);
  __ relocate(rspec);
  __ load_absolute_address(table, _trot_table_addr);
#ifdef ASSERT
    assert(_trot_table_addr != nullptr, "Translate table address must be initialized by now");
    assert((p2i(_trot_table_addr) & (TROT_ALIGNMENT-1)) == 0, "Translate table alignment error");
    for (int i = 0; i < 256; i++) {
      assert(i == *((jshort*)(_trot_table_addr+2*i)), "trot_table[%d] = %d", i, *((jshort*)(_trot_table_addr+2*i)));
    }
    {
      Label L;
      __ load_const_optimized(Z_R0, StubRoutines::zarch::_trot_table_addr);
      __ z_cgr(table, Z_R0);  // safety net
      __ z_bre(L);
      __ stop("crc_table: external word relocation does not work for load_absolute_address", 0x33);
      __ bind(L);
    }
    {
      Label L;
      __ load_const_optimized(Z_R0, 0x0004000500060007UL);
      __ z_clg(Z_R0, Address(table, 8));  // safety net
      __ z_bre(L);
      __ z_lg(Z_R0, Address(table, 8));   // Load data from memory, we know the constant we compared against.
      __ stop("trot_table: address or contents seems to be messed up", 0x22);
      __ bind(L);
    }
#endif
}


/**
 *  trot_table[]
 */

jlong StubRoutines::zarch::_trot_table[TROT_COLUMN_SIZE] = {
    0x0000000100020003UL, 0x0004000500060007UL, 0x00080009000a000bUL, 0x000c000d000e000fUL,
    0x0010001100120013UL, 0x0014001500160017UL, 0x00180019001a001bUL, 0x001c001d001e001fUL,
    0x0020002100220023UL, 0x0024002500260027UL, 0x00280029002a002bUL, 0x002c002d002e002fUL,
    0x0030003100320033UL, 0x0034003500360037UL, 0x00380039003a003bUL, 0x003c003d003e003fUL,
    0x0040004100420043UL, 0x0044004500460047UL, 0x00480049004a004bUL, 0x004c004d004e004fUL,
    0x0050005100520053UL, 0x0054005500560057UL, 0x00580059005a005bUL, 0x005c005d005e005fUL,
    0x0060006100620063UL, 0x0064006500660067UL, 0x00680069006a006bUL, 0x006c006d006e006fUL,
    0x0070007100720073UL, 0x0074007500760077UL, 0x00780079007a007bUL, 0x007c007d007e007fUL,
    0x0080008100820083UL, 0x0084008500860087UL, 0x00880089008a008bUL, 0x008c008d008e008fUL,
    0x0090009100920093UL, 0x0094009500960097UL, 0x00980099009a009bUL, 0x009c009d009e009fUL,
    0x00a000a100a200a3UL, 0x00a400a500a600a7UL, 0x00a800a900aa00abUL, 0x00ac00ad00ae00afUL,
    0x00b000b100b200b3UL, 0x00b400b500b600b7UL, 0x00b800b900ba00bbUL, 0x00bc00bd00be00bfUL,
    0x00c000c100c200c3UL, 0x00c400c500c600c7UL, 0x00c800c900ca00cbUL, 0x00cc00cd00ce00cfUL,
    0x00d000d100d200d3UL, 0x00d400d500d600d7UL, 0x00d800d900da00dbUL, 0x00dc00dd00de00dfUL,
    0x00e000e100e200e3UL, 0x00e400e500e600e7UL, 0x00e800e900ea00ebUL, 0x00ec00ed00ee00efUL,
    0x00f000f100f200f3UL, 0x00f400f500f600f7UL, 0x00f800f900fa00fbUL, 0x00fc00fd00fe00ffUL
  };


juint StubRoutines::zarch::_crc_table[CRC32_TABLES][CRC32_COLUMN_SIZE] = {
    /* polyBits = 7976584769 0x00000001db710641L, shifted = 0xedb88320 */
    /* CRC32 table for single bytes, auto-generated.  DO NOT MODIFY!  */
    /* CRC32 table 0 for quad-bytes (little-endian), auto-generated. DO NOT MODIFY! */
    {
      /* 0 */    0x00000000U, 0x77073096U, 0xee0e612cU, 0x990951baU, 0x076dc419U, 0x706af48fU, 0xe963a535U, 0x9e6495a3U
      /* 8 */  , 0x0edb8832U, 0x79dcb8a4U, 0xe0d5e91eU, 0x97d2d988U, 0x09b64c2bU, 0x7eb17cbdU, 0xe7b82d07U, 0x90bf1d91U
      /* 16 */  , 0x1db71064U, 0x6ab020f2U, 0xf3b97148U, 0x84be41deU, 0x1adad47dU, 0x6ddde4ebU, 0xf4d4b551U, 0x83d385c7U
      /* 24 */  , 0x136c9856U, 0x646ba8c0U, 0xfd62f97aU, 0x8a65c9ecU, 0x14015c4fU, 0x63066cd9U, 0xfa0f3d63U, 0x8d080df5U
      /* 32 */  , 0x3b6e20c8U, 0x4c69105eU, 0xd56041e4U, 0xa2677172U, 0x3c03e4d1U, 0x4b04d447U, 0xd20d85fdU, 0xa50ab56bU
      /* 40 */  , 0x35b5a8faU, 0x42b2986cU, 0xdbbbc9d6U, 0xacbcf940U, 0x32d86ce3U, 0x45df5c75U, 0xdcd60dcfU, 0xabd13d59U
      /* 48 */  , 0x26d930acU, 0x51de003aU, 0xc8d75180U, 0xbfd06116U, 0x21b4f4b5U, 0x56b3c423U, 0xcfba9599U, 0xb8bda50fU
      /* 56 */  , 0x2802b89eU, 0x5f058808U, 0xc60cd9b2U, 0xb10be924U, 0x2f6f7c87U, 0x58684c11U, 0xc1611dabU, 0xb6662d3dU
      /* 64 */  , 0x76dc4190U, 0x01db7106U, 0x98d220bcU, 0xefd5102aU, 0x71b18589U, 0x06b6b51fU, 0x9fbfe4a5U, 0xe8b8d433U
      /* 72 */  , 0x7807c9a2U, 0x0f00f934U, 0x9609a88eU, 0xe10e9818U, 0x7f6a0dbbU, 0x086d3d2dU, 0x91646c97U, 0xe6635c01U
      /* 80 */  , 0x6b6b51f4U, 0x1c6c6162U, 0x856530d8U, 0xf262004eU, 0x6c0695edU, 0x1b01a57bU, 0x8208f4c1U, 0xf50fc457U
      /* 88 */  , 0x65b0d9c6U, 0x12b7e950U, 0x8bbeb8eaU, 0xfcb9887cU, 0x62dd1ddfU, 0x15da2d49U, 0x8cd37cf3U, 0xfbd44c65U
      /* 96 */  , 0x4db26158U, 0x3ab551ceU, 0xa3bc0074U, 0xd4bb30e2U, 0x4adfa541U, 0x3dd895d7U, 0xa4d1c46dU, 0xd3d6f4fbU
      /* 104 */  , 0x4369e96aU, 0x346ed9fcU, 0xad678846U, 0xda60b8d0U, 0x44042d73U, 0x33031de5U, 0xaa0a4c5fU, 0xdd0d7cc9U
      /* 112 */  , 0x5005713cU, 0x270241aaU, 0xbe0b1010U, 0xc90c2086U, 0x5768b525U, 0x206f85b3U, 0xb966d409U, 0xce61e49fU
      /* 120 */  , 0x5edef90eU, 0x29d9c998U, 0xb0d09822U, 0xc7d7a8b4U, 0x59b33d17U, 0x2eb40d81U, 0xb7bd5c3bU, 0xc0ba6cadU
      /* 128 */  , 0xedb88320U, 0x9abfb3b6U, 0x03b6e20cU, 0x74b1d29aU, 0xead54739U, 0x9dd277afU, 0x04db2615U, 0x73dc1683U
      /* 136 */  , 0xe3630b12U, 0x94643b84U, 0x0d6d6a3eU, 0x7a6a5aa8U, 0xe40ecf0bU, 0x9309ff9dU, 0x0a00ae27U, 0x7d079eb1U
      /* 144 */  , 0xf00f9344U, 0x8708a3d2U, 0x1e01f268U, 0x6906c2feU, 0xf762575dU, 0x806567cbU, 0x196c3671U, 0x6e6b06e7U
      /* 152 */  , 0xfed41b76U, 0x89d32be0U, 0x10da7a5aU, 0x67dd4accU, 0xf9b9df6fU, 0x8ebeeff9U, 0x17b7be43U, 0x60b08ed5U
      /* 160 */  , 0xd6d6a3e8U, 0xa1d1937eU, 0x38d8c2c4U, 0x4fdff252U, 0xd1bb67f1U, 0xa6bc5767U, 0x3fb506ddU, 0x48b2364bU
      /* 168 */  , 0xd80d2bdaU, 0xaf0a1b4cU, 0x36034af6U, 0x41047a60U, 0xdf60efc3U, 0xa867df55U, 0x316e8eefU, 0x4669be79U
      /* 176 */  , 0xcb61b38cU, 0xbc66831aU, 0x256fd2a0U, 0x5268e236U, 0xcc0c7795U, 0xbb0b4703U, 0x220216b9U, 0x5505262fU
      /* 184 */  , 0xc5ba3bbeU, 0xb2bd0b28U, 0x2bb45a92U, 0x5cb36a04U, 0xc2d7ffa7U, 0xb5d0cf31U, 0x2cd99e8bU, 0x5bdeae1dU
      /* 192 */  , 0x9b64c2b0U, 0xec63f226U, 0x756aa39cU, 0x026d930aU, 0x9c0906a9U, 0xeb0e363fU, 0x72076785U, 0x05005713U
      /* 200 */  , 0x95bf4a82U, 0xe2b87a14U, 0x7bb12baeU, 0x0cb61b38U, 0x92d28e9bU, 0xe5d5be0dU, 0x7cdcefb7U, 0x0bdbdf21U
      /* 208 */  , 0x86d3d2d4U, 0xf1d4e242U, 0x68ddb3f8U, 0x1fda836eU, 0x81be16cdU, 0xf6b9265bU, 0x6fb077e1U, 0x18b74777U
      /* 216 */  , 0x88085ae6U, 0xff0f6a70U, 0x66063bcaU, 0x11010b5cU, 0x8f659effU, 0xf862ae69U, 0x616bffd3U, 0x166ccf45U
      /* 224 */  , 0xa00ae278U, 0xd70dd2eeU, 0x4e048354U, 0x3903b3c2U, 0xa7672661U, 0xd06016f7U, 0x4969474dU, 0x3e6e77dbU
      /* 232 */  , 0xaed16a4aU, 0xd9d65adcU, 0x40df0b66U, 0x37d83bf0U, 0xa9bcae53U, 0xdebb9ec5U, 0x47b2cf7fU, 0x30b5ffe9U
      /* 240 */  , 0xbdbdf21cU, 0xcabac28aU, 0x53b39330U, 0x24b4a3a6U, 0xbad03605U, 0xcdd70693U, 0x54de5729U, 0x23d967bfU
      /* 248 */  , 0xb3667a2eU, 0xc4614ab8U, 0x5d681b02U, 0x2a6f2b94U, 0xb40bbe37U, 0xc30c8ea1U, 0x5a05df1bU, 0x2d02ef8dU
    }
  #ifdef  CRC32_BYFOUR
    ,
    /* CRC32 table 1 for quad-bytes (little-endian), auto-generated. DO NOT MODIFY! */
    {
      /* 0 */    0x00000000U, 0x191b3141U, 0x32366282U, 0x2b2d53c3U, 0x646cc504U, 0x7d77f445U, 0x565aa786U, 0x4f4196c7U
      /* 8 */  , 0xc8d98a08U, 0xd1c2bb49U, 0xfaefe88aU, 0xe3f4d9cbU, 0xacb54f0cU, 0xb5ae7e4dU, 0x9e832d8eU, 0x87981ccfU
      /* 16 */  , 0x4ac21251U, 0x53d92310U, 0x78f470d3U, 0x61ef4192U, 0x2eaed755U, 0x37b5e614U, 0x1c98b5d7U, 0x05838496U
      /* 24 */  , 0x821b9859U, 0x9b00a918U, 0xb02dfadbU, 0xa936cb9aU, 0xe6775d5dU, 0xff6c6c1cU, 0xd4413fdfU, 0xcd5a0e9eU
      /* 32 */  , 0x958424a2U, 0x8c9f15e3U, 0xa7b24620U, 0xbea97761U, 0xf1e8e1a6U, 0xe8f3d0e7U, 0xc3de8324U, 0xdac5b265U
      /* 40 */  , 0x5d5daeaaU, 0x44469febU, 0x6f6bcc28U, 0x7670fd69U, 0x39316baeU, 0x202a5aefU, 0x0b07092cU, 0x121c386dU
      /* 48 */  , 0xdf4636f3U, 0xc65d07b2U, 0xed705471U, 0xf46b6530U, 0xbb2af3f7U, 0xa231c2b6U, 0x891c9175U, 0x9007a034U
      /* 56 */  , 0x179fbcfbU, 0x0e848dbaU, 0x25a9de79U, 0x3cb2ef38U, 0x73f379ffU, 0x6ae848beU, 0x41c51b7dU, 0x58de2a3cU
      /* 64 */  , 0xf0794f05U, 0xe9627e44U, 0xc24f2d87U, 0xdb541cc6U, 0x94158a01U, 0x8d0ebb40U, 0xa623e883U, 0xbf38d9c2U
      /* 72 */  , 0x38a0c50dU, 0x21bbf44cU, 0x0a96a78fU, 0x138d96ceU, 0x5ccc0009U, 0x45d73148U, 0x6efa628bU, 0x77e153caU
      /* 80 */  , 0xbabb5d54U, 0xa3a06c15U, 0x888d3fd6U, 0x91960e97U, 0xded79850U, 0xc7cca911U, 0xece1fad2U, 0xf5facb93U
      /* 88 */  , 0x7262d75cU, 0x6b79e61dU, 0x4054b5deU, 0x594f849fU, 0x160e1258U, 0x0f152319U, 0x243870daU, 0x3d23419bU
      /* 96 */  , 0x65fd6ba7U, 0x7ce65ae6U, 0x57cb0925U, 0x4ed03864U, 0x0191aea3U, 0x188a9fe2U, 0x33a7cc21U, 0x2abcfd60U
      /* 104 */  , 0xad24e1afU, 0xb43fd0eeU, 0x9f12832dU, 0x8609b26cU, 0xc94824abU, 0xd05315eaU, 0xfb7e4629U, 0xe2657768U
      /* 112 */  , 0x2f3f79f6U, 0x362448b7U, 0x1d091b74U, 0x04122a35U, 0x4b53bcf2U, 0x52488db3U, 0x7965de70U, 0x607eef31U
      /* 120 */  , 0xe7e6f3feU, 0xfefdc2bfU, 0xd5d0917cU, 0xcccba03dU, 0x838a36faU, 0x9a9107bbU, 0xb1bc5478U, 0xa8a76539U
      /* 128 */  , 0x3b83984bU, 0x2298a90aU, 0x09b5fac9U, 0x10aecb88U, 0x5fef5d4fU, 0x46f46c0eU, 0x6dd93fcdU, 0x74c20e8cU
      /* 136 */  , 0xf35a1243U, 0xea412302U, 0xc16c70c1U, 0xd8774180U, 0x9736d747U, 0x8e2de606U, 0xa500b5c5U, 0xbc1b8484U
      /* 144 */  , 0x71418a1aU, 0x685abb5bU, 0x4377e898U, 0x5a6cd9d9U, 0x152d4f1eU, 0x0c367e5fU, 0x271b2d9cU, 0x3e001cddU
      /* 152 */  , 0xb9980012U, 0xa0833153U, 0x8bae6290U, 0x92b553d1U, 0xddf4c516U, 0xc4eff457U, 0xefc2a794U, 0xf6d996d5U
      /* 160 */  , 0xae07bce9U, 0xb71c8da8U, 0x9c31de6bU, 0x852aef2aU, 0xca6b79edU, 0xd37048acU, 0xf85d1b6fU, 0xe1462a2eU
      /* 168 */  , 0x66de36e1U, 0x7fc507a0U, 0x54e85463U, 0x4df36522U, 0x02b2f3e5U, 0x1ba9c2a4U, 0x30849167U, 0x299fa026U
      /* 176 */  , 0xe4c5aeb8U, 0xfdde9ff9U, 0xd6f3cc3aU, 0xcfe8fd7bU, 0x80a96bbcU, 0x99b25afdU, 0xb29f093eU, 0xab84387fU
      /* 184 */  , 0x2c1c24b0U, 0x350715f1U, 0x1e2a4632U, 0x07317773U, 0x4870e1b4U, 0x516bd0f5U, 0x7a468336U, 0x635db277U
      /* 192 */  , 0xcbfad74eU, 0xd2e1e60fU, 0xf9ccb5ccU, 0xe0d7848dU, 0xaf96124aU, 0xb68d230bU, 0x9da070c8U, 0x84bb4189U
      /* 200 */  , 0x03235d46U, 0x1a386c07U, 0x31153fc4U, 0x280e0e85U, 0x674f9842U, 0x7e54a903U, 0x5579fac0U, 0x4c62cb81U
      /* 208 */  , 0x8138c51fU, 0x9823f45eU, 0xb30ea79dU, 0xaa1596dcU, 0xe554001bU, 0xfc4f315aU, 0xd7626299U, 0xce7953d8U
      /* 216 */  , 0x49e14f17U, 0x50fa7e56U, 0x7bd72d95U, 0x62cc1cd4U, 0x2d8d8a13U, 0x3496bb52U, 0x1fbbe891U, 0x06a0d9d0U
      /* 224 */  , 0x5e7ef3ecU, 0x4765c2adU, 0x6c48916eU, 0x7553a02fU, 0x3a1236e8U, 0x230907a9U, 0x0824546aU, 0x113f652bU
      /* 232 */  , 0x96a779e4U, 0x8fbc48a5U, 0xa4911b66U, 0xbd8a2a27U, 0xf2cbbce0U, 0xebd08da1U, 0xc0fdde62U, 0xd9e6ef23U
      /* 240 */  , 0x14bce1bdU, 0x0da7d0fcU, 0x268a833fU, 0x3f91b27eU, 0x70d024b9U, 0x69cb15f8U, 0x42e6463bU, 0x5bfd777aU
      /* 248 */  , 0xdc656bb5U, 0xc57e5af4U, 0xee530937U, 0xf7483876U, 0xb809aeb1U, 0xa1129ff0U, 0x8a3fcc33U, 0x9324fd72U
    }
    ,
    /* CRC32 table 2 for quad-bytes (little-endian), auto-generated. DO NOT MODIFY! */
    {
      /* 0 */    0x00000000U, 0x01c26a37U, 0x0384d46eU, 0x0246be59U, 0x0709a8dcU, 0x06cbc2ebU, 0x048d7cb2U, 0x054f1685U
      /* 8 */  , 0x0e1351b8U, 0x0fd13b8fU, 0x0d9785d6U, 0x0c55efe1U, 0x091af964U, 0x08d89353U, 0x0a9e2d0aU, 0x0b5c473dU
      /* 16 */  , 0x1c26a370U, 0x1de4c947U, 0x1fa2771eU, 0x1e601d29U, 0x1b2f0bacU, 0x1aed619bU, 0x18abdfc2U, 0x1969b5f5U
      /* 24 */  , 0x1235f2c8U, 0x13f798ffU, 0x11b126a6U, 0x10734c91U, 0x153c5a14U, 0x14fe3023U, 0x16b88e7aU, 0x177ae44dU
      /* 32 */  , 0x384d46e0U, 0x398f2cd7U, 0x3bc9928eU, 0x3a0bf8b9U, 0x3f44ee3cU, 0x3e86840bU, 0x3cc03a52U, 0x3d025065U
      /* 40 */  , 0x365e1758U, 0x379c7d6fU, 0x35dac336U, 0x3418a901U, 0x3157bf84U, 0x3095d5b3U, 0x32d36beaU, 0x331101ddU
      /* 48 */  , 0x246be590U, 0x25a98fa7U, 0x27ef31feU, 0x262d5bc9U, 0x23624d4cU, 0x22a0277bU, 0x20e69922U, 0x2124f315U
      /* 56 */  , 0x2a78b428U, 0x2bbade1fU, 0x29fc6046U, 0x283e0a71U, 0x2d711cf4U, 0x2cb376c3U, 0x2ef5c89aU, 0x2f37a2adU
      /* 64 */  , 0x709a8dc0U, 0x7158e7f7U, 0x731e59aeU, 0x72dc3399U, 0x7793251cU, 0x76514f2bU, 0x7417f172U, 0x75d59b45U
      /* 72 */  , 0x7e89dc78U, 0x7f4bb64fU, 0x7d0d0816U, 0x7ccf6221U, 0x798074a4U, 0x78421e93U, 0x7a04a0caU, 0x7bc6cafdU
      /* 80 */  , 0x6cbc2eb0U, 0x6d7e4487U, 0x6f38fadeU, 0x6efa90e9U, 0x6bb5866cU, 0x6a77ec5bU, 0x68315202U, 0x69f33835U
      /* 88 */  , 0x62af7f08U, 0x636d153fU, 0x612bab66U, 0x60e9c151U, 0x65a6d7d4U, 0x6464bde3U, 0x662203baU, 0x67e0698dU
      /* 96 */  , 0x48d7cb20U, 0x4915a117U, 0x4b531f4eU, 0x4a917579U, 0x4fde63fcU, 0x4e1c09cbU, 0x4c5ab792U, 0x4d98dda5U
      /* 104 */  , 0x46c49a98U, 0x4706f0afU, 0x45404ef6U, 0x448224c1U, 0x41cd3244U, 0x400f5873U, 0x4249e62aU, 0x438b8c1dU
      /* 112 */  , 0x54f16850U, 0x55330267U, 0x5775bc3eU, 0x56b7d609U, 0x53f8c08cU, 0x523aaabbU, 0x507c14e2U, 0x51be7ed5U
      /* 120 */  , 0x5ae239e8U, 0x5b2053dfU, 0x5966ed86U, 0x58a487b1U, 0x5deb9134U, 0x5c29fb03U, 0x5e6f455aU, 0x5fad2f6dU
      /* 128 */  , 0xe1351b80U, 0xe0f771b7U, 0xe2b1cfeeU, 0xe373a5d9U, 0xe63cb35cU, 0xe7fed96bU, 0xe5b86732U, 0xe47a0d05U
      /* 136 */  , 0xef264a38U, 0xeee4200fU, 0xeca29e56U, 0xed60f461U, 0xe82fe2e4U, 0xe9ed88d3U, 0xebab368aU, 0xea695cbdU
      /* 144 */  , 0xfd13b8f0U, 0xfcd1d2c7U, 0xfe976c9eU, 0xff5506a9U, 0xfa1a102cU, 0xfbd87a1bU, 0xf99ec442U, 0xf85cae75U
      /* 152 */  , 0xf300e948U, 0xf2c2837fU, 0xf0843d26U, 0xf1465711U, 0xf4094194U, 0xf5cb2ba3U, 0xf78d95faU, 0xf64fffcdU
      /* 160 */  , 0xd9785d60U, 0xd8ba3757U, 0xdafc890eU, 0xdb3ee339U, 0xde71f5bcU, 0xdfb39f8bU, 0xddf521d2U, 0xdc374be5U
      /* 168 */  , 0xd76b0cd8U, 0xd6a966efU, 0xd4efd8b6U, 0xd52db281U, 0xd062a404U, 0xd1a0ce33U, 0xd3e6706aU, 0xd2241a5dU
      /* 176 */  , 0xc55efe10U, 0xc49c9427U, 0xc6da2a7eU, 0xc7184049U, 0xc25756ccU, 0xc3953cfbU, 0xc1d382a2U, 0xc011e895U
      /* 184 */  , 0xcb4dafa8U, 0xca8fc59fU, 0xc8c97bc6U, 0xc90b11f1U, 0xcc440774U, 0xcd866d43U, 0xcfc0d31aU, 0xce02b92dU
      /* 192 */  , 0x91af9640U, 0x906dfc77U, 0x922b422eU, 0x93e92819U, 0x96a63e9cU, 0x976454abU, 0x9522eaf2U, 0x94e080c5U
      /* 200 */  , 0x9fbcc7f8U, 0x9e7eadcfU, 0x9c381396U, 0x9dfa79a1U, 0x98b56f24U, 0x99770513U, 0x9b31bb4aU, 0x9af3d17dU
      /* 208 */  , 0x8d893530U, 0x8c4b5f07U, 0x8e0de15eU, 0x8fcf8b69U, 0x8a809decU, 0x8b42f7dbU, 0x89044982U, 0x88c623b5U
      /* 216 */  , 0x839a6488U, 0x82580ebfU, 0x801eb0e6U, 0x81dcdad1U, 0x8493cc54U, 0x8551a663U, 0x8717183aU, 0x86d5720dU
      /* 224 */  , 0xa9e2d0a0U, 0xa820ba97U, 0xaa6604ceU, 0xaba46ef9U, 0xaeeb787cU, 0xaf29124bU, 0xad6fac12U, 0xacadc625U
      /* 232 */  , 0xa7f18118U, 0xa633eb2fU, 0xa4755576U, 0xa5b73f41U, 0xa0f829c4U, 0xa13a43f3U, 0xa37cfdaaU, 0xa2be979dU
      /* 240 */  , 0xb5c473d0U, 0xb40619e7U, 0xb640a7beU, 0xb782cd89U, 0xb2cddb0cU, 0xb30fb13bU, 0xb1490f62U, 0xb08b6555U
      /* 248 */  , 0xbbd72268U, 0xba15485fU, 0xb853f606U, 0xb9919c31U, 0xbcde8ab4U, 0xbd1ce083U, 0xbf5a5edaU, 0xbe9834edU
    }
    ,
    /* CRC32 table 3 for quad-bytes (little-endian), auto-generated. DO NOT MODIFY! */
    {
      /* 0 */    0x00000000U, 0xb8bc6765U, 0xaa09c88bU, 0x12b5afeeU, 0x8f629757U, 0x37def032U, 0x256b5fdcU, 0x9dd738b9U
      /* 8 */  , 0xc5b428efU, 0x7d084f8aU, 0x6fbde064U, 0xd7018701U, 0x4ad6bfb8U, 0xf26ad8ddU, 0xe0df7733U, 0x58631056U
      /* 16 */  , 0x5019579fU, 0xe8a530faU, 0xfa109f14U, 0x42acf871U, 0xdf7bc0c8U, 0x67c7a7adU, 0x75720843U, 0xcdce6f26U
      /* 24 */  , 0x95ad7f70U, 0x2d111815U, 0x3fa4b7fbU, 0x8718d09eU, 0x1acfe827U, 0xa2738f42U, 0xb0c620acU, 0x087a47c9U
      /* 32 */  , 0xa032af3eU, 0x188ec85bU, 0x0a3b67b5U, 0xb28700d0U, 0x2f503869U, 0x97ec5f0cU, 0x8559f0e2U, 0x3de59787U
      /* 40 */  , 0x658687d1U, 0xdd3ae0b4U, 0xcf8f4f5aU, 0x7733283fU, 0xeae41086U, 0x525877e3U, 0x40edd80dU, 0xf851bf68U
      /* 48 */  , 0xf02bf8a1U, 0x48979fc4U, 0x5a22302aU, 0xe29e574fU, 0x7f496ff6U, 0xc7f50893U, 0xd540a77dU, 0x6dfcc018U
      /* 56 */  , 0x359fd04eU, 0x8d23b72bU, 0x9f9618c5U, 0x272a7fa0U, 0xbafd4719U, 0x0241207cU, 0x10f48f92U, 0xa848e8f7U
      /* 64 */  , 0x9b14583dU, 0x23a83f58U, 0x311d90b6U, 0x89a1f7d3U, 0x1476cf6aU, 0xaccaa80fU, 0xbe7f07e1U, 0x06c36084U
      /* 72 */  , 0x5ea070d2U, 0xe61c17b7U, 0xf4a9b859U, 0x4c15df3cU, 0xd1c2e785U, 0x697e80e0U, 0x7bcb2f0eU, 0xc377486bU
      /* 80 */  , 0xcb0d0fa2U, 0x73b168c7U, 0x6104c729U, 0xd9b8a04cU, 0x446f98f5U, 0xfcd3ff90U, 0xee66507eU, 0x56da371bU
      /* 88 */  , 0x0eb9274dU, 0xb6054028U, 0xa4b0efc6U, 0x1c0c88a3U, 0x81dbb01aU, 0x3967d77fU, 0x2bd27891U, 0x936e1ff4U
      /* 96 */  , 0x3b26f703U, 0x839a9066U, 0x912f3f88U, 0x299358edU, 0xb4446054U, 0x0cf80731U, 0x1e4da8dfU, 0xa6f1cfbaU
      /* 104 */  , 0xfe92dfecU, 0x462eb889U, 0x549b1767U, 0xec277002U, 0x71f048bbU, 0xc94c2fdeU, 0xdbf98030U, 0x6345e755U
      /* 112 */  , 0x6b3fa09cU, 0xd383c7f9U, 0xc1366817U, 0x798a0f72U, 0xe45d37cbU, 0x5ce150aeU, 0x4e54ff40U, 0xf6e89825U
      /* 120 */  , 0xae8b8873U, 0x1637ef16U, 0x048240f8U, 0xbc3e279dU, 0x21e91f24U, 0x99557841U, 0x8be0d7afU, 0x335cb0caU
      /* 128 */  , 0xed59b63bU, 0x55e5d15eU, 0x47507eb0U, 0xffec19d5U, 0x623b216cU, 0xda874609U, 0xc832e9e7U, 0x708e8e82U
      /* 136 */  , 0x28ed9ed4U, 0x9051f9b1U, 0x82e4565fU, 0x3a58313aU, 0xa78f0983U, 0x1f336ee6U, 0x0d86c108U, 0xb53aa66dU
      /* 144 */  , 0xbd40e1a4U, 0x05fc86c1U, 0x1749292fU, 0xaff54e4aU, 0x322276f3U, 0x8a9e1196U, 0x982bbe78U, 0x2097d91dU
      /* 152 */  , 0x78f4c94bU, 0xc048ae2eU, 0xd2fd01c0U, 0x6a4166a5U, 0xf7965e1cU, 0x4f2a3979U, 0x5d9f9697U, 0xe523f1f2U
      /* 160 */  , 0x4d6b1905U, 0xf5d77e60U, 0xe762d18eU, 0x5fdeb6ebU, 0xc2098e52U, 0x7ab5e937U, 0x680046d9U, 0xd0bc21bcU
      /* 168 */  , 0x88df31eaU, 0x3063568fU, 0x22d6f961U, 0x9a6a9e04U, 0x07bda6bdU, 0xbf01c1d8U, 0xadb46e36U, 0x15080953U
      /* 176 */  , 0x1d724e9aU, 0xa5ce29ffU, 0xb77b8611U, 0x0fc7e174U, 0x9210d9cdU, 0x2aacbea8U, 0x38191146U, 0x80a57623U
      /* 184 */  , 0xd8c66675U, 0x607a0110U, 0x72cfaefeU, 0xca73c99bU, 0x57a4f122U, 0xef189647U, 0xfdad39a9U, 0x45115eccU
      /* 192 */  , 0x764dee06U, 0xcef18963U, 0xdc44268dU, 0x64f841e8U, 0xf92f7951U, 0x41931e34U, 0x5326b1daU, 0xeb9ad6bfU
      /* 200 */  , 0xb3f9c6e9U, 0x0b45a18cU, 0x19f00e62U, 0xa14c6907U, 0x3c9b51beU, 0x842736dbU, 0x96929935U, 0x2e2efe50U
      /* 208 */  , 0x2654b999U, 0x9ee8defcU, 0x8c5d7112U, 0x34e11677U, 0xa9362eceU, 0x118a49abU, 0x033fe645U, 0xbb838120U
      /* 216 */  , 0xe3e09176U, 0x5b5cf613U, 0x49e959fdU, 0xf1553e98U, 0x6c820621U, 0xd43e6144U, 0xc68bceaaU, 0x7e37a9cfU
      /* 224 */  , 0xd67f4138U, 0x6ec3265dU, 0x7c7689b3U, 0xc4caeed6U, 0x591dd66fU, 0xe1a1b10aU, 0xf3141ee4U, 0x4ba87981U
      /* 232 */  , 0x13cb69d7U, 0xab770eb2U, 0xb9c2a15cU, 0x017ec639U, 0x9ca9fe80U, 0x241599e5U, 0x36a0360bU, 0x8e1c516eU
      /* 240 */  , 0x866616a7U, 0x3eda71c2U, 0x2c6fde2cU, 0x94d3b949U, 0x090481f0U, 0xb1b8e695U, 0xa30d497bU, 0x1bb12e1eU
      /* 248 */  , 0x43d23e48U, 0xfb6e592dU, 0xe9dbf6c3U, 0x516791a6U, 0xccb0a91fU, 0x740cce7aU, 0x66b96194U, 0xde0506f1U
    }
    ,
    /* CRC32 table 4 for quad-bytes (  big-endian ), auto-generated. DO NOT MODIFY! */
    {
      /* 0 */    0x00000000U, 0x96300777U, 0x2c610eeeU, 0xba510999U, 0x19c46d07U, 0x8ff46a70U, 0x35a563e9U, 0xa395649eU
      /* 8 */  , 0x3288db0eU, 0xa4b8dc79U, 0x1ee9d5e0U, 0x88d9d297U, 0x2b4cb609U, 0xbd7cb17eU, 0x072db8e7U, 0x911dbf90U
      /* 16 */  , 0x6410b71dU, 0xf220b06aU, 0x4871b9f3U, 0xde41be84U, 0x7dd4da1aU, 0xebe4dd6dU, 0x51b5d4f4U, 0xc785d383U
      /* 24 */  , 0x56986c13U, 0xc0a86b64U, 0x7af962fdU, 0xecc9658aU, 0x4f5c0114U, 0xd96c0663U, 0x633d0ffaU, 0xf50d088dU
      /* 32 */  , 0xc8206e3bU, 0x5e10694cU, 0xe44160d5U, 0x727167a2U, 0xd1e4033cU, 0x47d4044bU, 0xfd850dd2U, 0x6bb50aa5U
      /* 40 */  , 0xfaa8b535U, 0x6c98b242U, 0xd6c9bbdbU, 0x40f9bcacU, 0xe36cd832U, 0x755cdf45U, 0xcf0dd6dcU, 0x593dd1abU
      /* 48 */  , 0xac30d926U, 0x3a00de51U, 0x8051d7c8U, 0x1661d0bfU, 0xb5f4b421U, 0x23c4b356U, 0x9995bacfU, 0x0fa5bdb8U
      /* 56 */  , 0x9eb80228U, 0x0888055fU, 0xb2d90cc6U, 0x24e90bb1U, 0x877c6f2fU, 0x114c6858U, 0xab1d61c1U, 0x3d2d66b6U
      /* 64 */  , 0x9041dc76U, 0x0671db01U, 0xbc20d298U, 0x2a10d5efU, 0x8985b171U, 0x1fb5b606U, 0xa5e4bf9fU, 0x33d4b8e8U
      /* 72 */  , 0xa2c90778U, 0x34f9000fU, 0x8ea80996U, 0x18980ee1U, 0xbb0d6a7fU, 0x2d3d6d08U, 0x976c6491U, 0x015c63e6U
      /* 80 */  , 0xf4516b6bU, 0x62616c1cU, 0xd8306585U, 0x4e0062f2U, 0xed95066cU, 0x7ba5011bU, 0xc1f40882U, 0x57c40ff5U
      /* 88 */  , 0xc6d9b065U, 0x50e9b712U, 0xeab8be8bU, 0x7c88b9fcU, 0xdf1ddd62U, 0x492dda15U, 0xf37cd38cU, 0x654cd4fbU
      /* 96 */  , 0x5861b24dU, 0xce51b53aU, 0x7400bca3U, 0xe230bbd4U, 0x41a5df4aU, 0xd795d83dU, 0x6dc4d1a4U, 0xfbf4d6d3U
      /* 104 */  , 0x6ae96943U, 0xfcd96e34U, 0x468867adU, 0xd0b860daU, 0x732d0444U, 0xe51d0333U, 0x5f4c0aaaU, 0xc97c0dddU
      /* 112 */  , 0x3c710550U, 0xaa410227U, 0x10100bbeU, 0x86200cc9U, 0x25b56857U, 0xb3856f20U, 0x09d466b9U, 0x9fe461ceU
      /* 120 */  , 0x0ef9de5eU, 0x98c9d929U, 0x2298d0b0U, 0xb4a8d7c7U, 0x173db359U, 0x810db42eU, 0x3b5cbdb7U, 0xad6cbac0U
      /* 128 */  , 0x2083b8edU, 0xb6b3bf9aU, 0x0ce2b603U, 0x9ad2b174U, 0x3947d5eaU, 0xaf77d29dU, 0x1526db04U, 0x8316dc73U
      /* 136 */  , 0x120b63e3U, 0x843b6494U, 0x3e6a6d0dU, 0xa85a6a7aU, 0x0bcf0ee4U, 0x9dff0993U, 0x27ae000aU, 0xb19e077dU
      /* 144 */  , 0x44930ff0U, 0xd2a30887U, 0x68f2011eU, 0xfec20669U, 0x5d5762f7U, 0xcb676580U, 0x71366c19U, 0xe7066b6eU
      /* 152 */  , 0x761bd4feU, 0xe02bd389U, 0x5a7ada10U, 0xcc4add67U, 0x6fdfb9f9U, 0xf9efbe8eU, 0x43beb717U, 0xd58eb060U
      /* 160 */  , 0xe8a3d6d6U, 0x7e93d1a1U, 0xc4c2d838U, 0x52f2df4fU, 0xf167bbd1U, 0x6757bca6U, 0xdd06b53fU, 0x4b36b248U
      /* 168 */  , 0xda2b0dd8U, 0x4c1b0aafU, 0xf64a0336U, 0x607a0441U, 0xc3ef60dfU, 0x55df67a8U, 0xef8e6e31U, 0x79be6946U
      /* 176 */  , 0x8cb361cbU, 0x1a8366bcU, 0xa0d26f25U, 0x36e26852U, 0x95770cccU, 0x03470bbbU, 0xb9160222U, 0x2f260555U
      /* 184 */  , 0xbe3bbac5U, 0x280bbdb2U, 0x925ab42bU, 0x046ab35cU, 0xa7ffd7c2U, 0x31cfd0b5U, 0x8b9ed92cU, 0x1daede5bU
      /* 192 */  , 0xb0c2649bU, 0x26f263ecU, 0x9ca36a75U, 0x0a936d02U, 0xa906099cU, 0x3f360eebU, 0x85670772U, 0x13570005U
      /* 200 */  , 0x824abf95U, 0x147ab8e2U, 0xae2bb17bU, 0x381bb60cU, 0x9b8ed292U, 0x0dbed5e5U, 0xb7efdc7cU, 0x21dfdb0bU
      /* 208 */  , 0xd4d2d386U, 0x42e2d4f1U, 0xf8b3dd68U, 0x6e83da1fU, 0xcd16be81U, 0x5b26b9f6U, 0xe177b06fU, 0x7747b718U
      /* 216 */  , 0xe65a0888U, 0x706a0fffU, 0xca3b0666U, 0x5c0b0111U, 0xff9e658fU, 0x69ae62f8U, 0xd3ff6b61U, 0x45cf6c16U
      /* 224 */  , 0x78e20aa0U, 0xeed20dd7U, 0x5483044eU, 0xc2b30339U, 0x612667a7U, 0xf71660d0U, 0x4d476949U, 0xdb776e3eU
      /* 232 */  , 0x4a6ad1aeU, 0xdc5ad6d9U, 0x660bdf40U, 0xf03bd837U, 0x53aebca9U, 0xc59ebbdeU, 0x7fcfb247U, 0xe9ffb530U
      /* 240 */  , 0x1cf2bdbdU, 0x8ac2bacaU, 0x3093b353U, 0xa6a3b424U, 0x0536d0baU, 0x9306d7cdU, 0x2957de54U, 0xbf67d923U
      /* 248 */  , 0x2e7a66b3U, 0xb84a61c4U, 0x021b685dU, 0x942b6f2aU, 0x37be0bb4U, 0xa18e0cc3U, 0x1bdf055aU, 0x8def022dU
    }
    ,
    /* CRC32 table 5 for quad-bytes (  big-endian ), auto-generated. DO NOT MODIFY! */
    {
      /* 0 */    0x00000000U, 0x41311b19U, 0x82623632U, 0xc3532d2bU, 0x04c56c64U, 0x45f4777dU, 0x86a75a56U, 0xc796414fU
      /* 8 */  , 0x088ad9c8U, 0x49bbc2d1U, 0x8ae8effaU, 0xcbd9f4e3U, 0x0c4fb5acU, 0x4d7eaeb5U, 0x8e2d839eU, 0xcf1c9887U
      /* 16 */  , 0x5112c24aU, 0x1023d953U, 0xd370f478U, 0x9241ef61U, 0x55d7ae2eU, 0x14e6b537U, 0xd7b5981cU, 0x96848305U
      /* 24 */  , 0x59981b82U, 0x18a9009bU, 0xdbfa2db0U, 0x9acb36a9U, 0x5d5d77e6U, 0x1c6c6cffU, 0xdf3f41d4U, 0x9e0e5acdU
      /* 32 */  , 0xa2248495U, 0xe3159f8cU, 0x2046b2a7U, 0x6177a9beU, 0xa6e1e8f1U, 0xe7d0f3e8U, 0x2483dec3U, 0x65b2c5daU
      /* 40 */  , 0xaaae5d5dU, 0xeb9f4644U, 0x28cc6b6fU, 0x69fd7076U, 0xae6b3139U, 0xef5a2a20U, 0x2c09070bU, 0x6d381c12U
      /* 48 */  , 0xf33646dfU, 0xb2075dc6U, 0x715470edU, 0x30656bf4U, 0xf7f32abbU, 0xb6c231a2U, 0x75911c89U, 0x34a00790U
      /* 56 */  , 0xfbbc9f17U, 0xba8d840eU, 0x79dea925U, 0x38efb23cU, 0xff79f373U, 0xbe48e86aU, 0x7d1bc541U, 0x3c2ade58U
      /* 64 */  , 0x054f79f0U, 0x447e62e9U, 0x872d4fc2U, 0xc61c54dbU, 0x018a1594U, 0x40bb0e8dU, 0x83e823a6U, 0xc2d938bfU
      /* 72 */  , 0x0dc5a038U, 0x4cf4bb21U, 0x8fa7960aU, 0xce968d13U, 0x0900cc5cU, 0x4831d745U, 0x8b62fa6eU, 0xca53e177U
      /* 80 */  , 0x545dbbbaU, 0x156ca0a3U, 0xd63f8d88U, 0x970e9691U, 0x5098d7deU, 0x11a9ccc7U, 0xd2fae1ecU, 0x93cbfaf5U
      /* 88 */  , 0x5cd76272U, 0x1de6796bU, 0xdeb55440U, 0x9f844f59U, 0x58120e16U, 0x1923150fU, 0xda703824U, 0x9b41233dU
      /* 96 */  , 0xa76bfd65U, 0xe65ae67cU, 0x2509cb57U, 0x6438d04eU, 0xa3ae9101U, 0xe29f8a18U, 0x21cca733U, 0x60fdbc2aU
      /* 104 */  , 0xafe124adU, 0xeed03fb4U, 0x2d83129fU, 0x6cb20986U, 0xab2448c9U, 0xea1553d0U, 0x29467efbU, 0x687765e2U
      /* 112 */  , 0xf6793f2fU, 0xb7482436U, 0x741b091dU, 0x352a1204U, 0xf2bc534bU, 0xb38d4852U, 0x70de6579U, 0x31ef7e60U
      /* 120 */  , 0xfef3e6e7U, 0xbfc2fdfeU, 0x7c91d0d5U, 0x3da0cbccU, 0xfa368a83U, 0xbb07919aU, 0x7854bcb1U, 0x3965a7a8U
      /* 128 */  , 0x4b98833bU, 0x0aa99822U, 0xc9fab509U, 0x88cbae10U, 0x4f5def5fU, 0x0e6cf446U, 0xcd3fd96dU, 0x8c0ec274U
      /* 136 */  , 0x43125af3U, 0x022341eaU, 0xc1706cc1U, 0x804177d8U, 0x47d73697U, 0x06e62d8eU, 0xc5b500a5U, 0x84841bbcU
      /* 144 */  , 0x1a8a4171U, 0x5bbb5a68U, 0x98e87743U, 0xd9d96c5aU, 0x1e4f2d15U, 0x5f7e360cU, 0x9c2d1b27U, 0xdd1c003eU
      /* 152 */  , 0x120098b9U, 0x533183a0U, 0x9062ae8bU, 0xd153b592U, 0x16c5f4ddU, 0x57f4efc4U, 0x94a7c2efU, 0xd596d9f6U
      /* 160 */  , 0xe9bc07aeU, 0xa88d1cb7U, 0x6bde319cU, 0x2aef2a85U, 0xed796bcaU, 0xac4870d3U, 0x6f1b5df8U, 0x2e2a46e1U
      /* 168 */  , 0xe136de66U, 0xa007c57fU, 0x6354e854U, 0x2265f34dU, 0xe5f3b202U, 0xa4c2a91bU, 0x67918430U, 0x26a09f29U
      /* 176 */  , 0xb8aec5e4U, 0xf99fdefdU, 0x3accf3d6U, 0x7bfde8cfU, 0xbc6ba980U, 0xfd5ab299U, 0x3e099fb2U, 0x7f3884abU
      /* 184 */  , 0xb0241c2cU, 0xf1150735U, 0x32462a1eU, 0x73773107U, 0xb4e17048U, 0xf5d06b51U, 0x3683467aU, 0x77b25d63U
      /* 192 */  , 0x4ed7facbU, 0x0fe6e1d2U, 0xccb5ccf9U, 0x8d84d7e0U, 0x4a1296afU, 0x0b238db6U, 0xc870a09dU, 0x8941bb84U
      /* 200 */  , 0x465d2303U, 0x076c381aU, 0xc43f1531U, 0x850e0e28U, 0x42984f67U, 0x03a9547eU, 0xc0fa7955U, 0x81cb624cU
      /* 208 */  , 0x1fc53881U, 0x5ef42398U, 0x9da70eb3U, 0xdc9615aaU, 0x1b0054e5U, 0x5a314ffcU, 0x996262d7U, 0xd85379ceU
      /* 216 */  , 0x174fe149U, 0x567efa50U, 0x952dd77bU, 0xd41ccc62U, 0x138a8d2dU, 0x52bb9634U, 0x91e8bb1fU, 0xd0d9a006U
      /* 224 */  , 0xecf37e5eU, 0xadc26547U, 0x6e91486cU, 0x2fa05375U, 0xe836123aU, 0xa9070923U, 0x6a542408U, 0x2b653f11U
      /* 232 */  , 0xe479a796U, 0xa548bc8fU, 0x661b91a4U, 0x272a8abdU, 0xe0bccbf2U, 0xa18dd0ebU, 0x62defdc0U, 0x23efe6d9U
      /* 240 */  , 0xbde1bc14U, 0xfcd0a70dU, 0x3f838a26U, 0x7eb2913fU, 0xb924d070U, 0xf815cb69U, 0x3b46e642U, 0x7a77fd5bU
      /* 248 */  , 0xb56b65dcU, 0xf45a7ec5U, 0x370953eeU, 0x763848f7U, 0xb1ae09b8U, 0xf09f12a1U, 0x33cc3f8aU, 0x72fd2493U
    }
    ,
    /* CRC32 table 6 for quad-bytes (  big-endian ), auto-generated. DO NOT MODIFY! */
    {
      /* 0 */    0x00000000U, 0x376ac201U, 0x6ed48403U, 0x59be4602U, 0xdca80907U, 0xebc2cb06U, 0xb27c8d04U, 0x85164f05U
      /* 8 */  , 0xb851130eU, 0x8f3bd10fU, 0xd685970dU, 0xe1ef550cU, 0x64f91a09U, 0x5393d808U, 0x0a2d9e0aU, 0x3d475c0bU
      /* 16 */  , 0x70a3261cU, 0x47c9e41dU, 0x1e77a21fU, 0x291d601eU, 0xac0b2f1bU, 0x9b61ed1aU, 0xc2dfab18U, 0xf5b56919U
      /* 24 */  , 0xc8f23512U, 0xff98f713U, 0xa626b111U, 0x914c7310U, 0x145a3c15U, 0x2330fe14U, 0x7a8eb816U, 0x4de47a17U
      /* 32 */  , 0xe0464d38U, 0xd72c8f39U, 0x8e92c93bU, 0xb9f80b3aU, 0x3cee443fU, 0x0b84863eU, 0x523ac03cU, 0x6550023dU
      /* 40 */  , 0x58175e36U, 0x6f7d9c37U, 0x36c3da35U, 0x01a91834U, 0x84bf5731U, 0xb3d59530U, 0xea6bd332U, 0xdd011133U
      /* 48 */  , 0x90e56b24U, 0xa78fa925U, 0xfe31ef27U, 0xc95b2d26U, 0x4c4d6223U, 0x7b27a022U, 0x2299e620U, 0x15f32421U
      /* 56 */  , 0x28b4782aU, 0x1fdeba2bU, 0x4660fc29U, 0x710a3e28U, 0xf41c712dU, 0xc376b32cU, 0x9ac8f52eU, 0xada2372fU
      /* 64 */  , 0xc08d9a70U, 0xf7e75871U, 0xae591e73U, 0x9933dc72U, 0x1c259377U, 0x2b4f5176U, 0x72f11774U, 0x459bd575U
      /* 72 */  , 0x78dc897eU, 0x4fb64b7fU, 0x16080d7dU, 0x2162cf7cU, 0xa4748079U, 0x931e4278U, 0xcaa0047aU, 0xfdcac67bU
      /* 80 */  , 0xb02ebc6cU, 0x87447e6dU, 0xdefa386fU, 0xe990fa6eU, 0x6c86b56bU, 0x5bec776aU, 0x02523168U, 0x3538f369U
      /* 88 */  , 0x087faf62U, 0x3f156d63U, 0x66ab2b61U, 0x51c1e960U, 0xd4d7a665U, 0xe3bd6464U, 0xba032266U, 0x8d69e067U
      /* 96 */  , 0x20cbd748U, 0x17a11549U, 0x4e1f534bU, 0x7975914aU, 0xfc63de4fU, 0xcb091c4eU, 0x92b75a4cU, 0xa5dd984dU
      /* 104 */  , 0x989ac446U, 0xaff00647U, 0xf64e4045U, 0xc1248244U, 0x4432cd41U, 0x73580f40U, 0x2ae64942U, 0x1d8c8b43U
      /* 112 */  , 0x5068f154U, 0x67023355U, 0x3ebc7557U, 0x09d6b756U, 0x8cc0f853U, 0xbbaa3a52U, 0xe2147c50U, 0xd57ebe51U
      /* 120 */  , 0xe839e25aU, 0xdf53205bU, 0x86ed6659U, 0xb187a458U, 0x3491eb5dU, 0x03fb295cU, 0x5a456f5eU, 0x6d2fad5fU
      /* 128 */  , 0x801b35e1U, 0xb771f7e0U, 0xeecfb1e2U, 0xd9a573e3U, 0x5cb33ce6U, 0x6bd9fee7U, 0x3267b8e5U, 0x050d7ae4U
      /* 136 */  , 0x384a26efU, 0x0f20e4eeU, 0x569ea2ecU, 0x61f460edU, 0xe4e22fe8U, 0xd388ede9U, 0x8a36abebU, 0xbd5c69eaU
      /* 144 */  , 0xf0b813fdU, 0xc7d2d1fcU, 0x9e6c97feU, 0xa90655ffU, 0x2c101afaU, 0x1b7ad8fbU, 0x42c49ef9U, 0x75ae5cf8U
      /* 152 */  , 0x48e900f3U, 0x7f83c2f2U, 0x263d84f0U, 0x115746f1U, 0x944109f4U, 0xa32bcbf5U, 0xfa958df7U, 0xcdff4ff6U
      /* 160 */  , 0x605d78d9U, 0x5737bad8U, 0x0e89fcdaU, 0x39e33edbU, 0xbcf571deU, 0x8b9fb3dfU, 0xd221f5ddU, 0xe54b37dcU
      /* 168 */  , 0xd80c6bd7U, 0xef66a9d6U, 0xb6d8efd4U, 0x81b22dd5U, 0x04a462d0U, 0x33cea0d1U, 0x6a70e6d3U, 0x5d1a24d2U
      /* 176 */  , 0x10fe5ec5U, 0x27949cc4U, 0x7e2adac6U, 0x494018c7U, 0xcc5657c2U, 0xfb3c95c3U, 0xa282d3c1U, 0x95e811c0U
      /* 184 */  , 0xa8af4dcbU, 0x9fc58fcaU, 0xc67bc9c8U, 0xf1110bc9U, 0x740744ccU, 0x436d86cdU, 0x1ad3c0cfU, 0x2db902ceU
      /* 192 */  , 0x4096af91U, 0x77fc6d90U, 0x2e422b92U, 0x1928e993U, 0x9c3ea696U, 0xab546497U, 0xf2ea2295U, 0xc580e094U
      /* 200 */  , 0xf8c7bc9fU, 0xcfad7e9eU, 0x9613389cU, 0xa179fa9dU, 0x246fb598U, 0x13057799U, 0x4abb319bU, 0x7dd1f39aU
      /* 208 */  , 0x3035898dU, 0x075f4b8cU, 0x5ee10d8eU, 0x698bcf8fU, 0xec9d808aU, 0xdbf7428bU, 0x82490489U, 0xb523c688U
      /* 216 */  , 0x88649a83U, 0xbf0e5882U, 0xe6b01e80U, 0xd1dadc81U, 0x54cc9384U, 0x63a65185U, 0x3a181787U, 0x0d72d586U
      /* 224 */  , 0xa0d0e2a9U, 0x97ba20a8U, 0xce0466aaU, 0xf96ea4abU, 0x7c78ebaeU, 0x4b1229afU, 0x12ac6fadU, 0x25c6adacU
      /* 232 */  , 0x1881f1a7U, 0x2feb33a6U, 0x765575a4U, 0x413fb7a5U, 0xc429f8a0U, 0xf3433aa1U, 0xaafd7ca3U, 0x9d97bea2U
      /* 240 */  , 0xd073c4b5U, 0xe71906b4U, 0xbea740b6U, 0x89cd82b7U, 0x0cdbcdb2U, 0x3bb10fb3U, 0x620f49b1U, 0x55658bb0U
      /* 248 */  , 0x6822d7bbU, 0x5f4815baU, 0x06f653b8U, 0x319c91b9U, 0xb48adebcU, 0x83e01cbdU, 0xda5e5abfU, 0xed3498beU
    }
    ,
    /* CRC32 table 7 for quad-bytes (  big-endian ), auto-generated. DO NOT MODIFY! */
    {
      /* 0 */    0x00000000U, 0x6567bcb8U, 0x8bc809aaU, 0xeeafb512U, 0x5797628fU, 0x32f0de37U, 0xdc5f6b25U, 0xb938d79dU
      /* 8 */  , 0xef28b4c5U, 0x8a4f087dU, 0x64e0bd6fU, 0x018701d7U, 0xb8bfd64aU, 0xddd86af2U, 0x3377dfe0U, 0x56106358U
      /* 16 */  , 0x9f571950U, 0xfa30a5e8U, 0x149f10faU, 0x71f8ac42U, 0xc8c07bdfU, 0xada7c767U, 0x43087275U, 0x266fcecdU
      /* 24 */  , 0x707fad95U, 0x1518112dU, 0xfbb7a43fU, 0x9ed01887U, 0x27e8cf1aU, 0x428f73a2U, 0xac20c6b0U, 0xc9477a08U
      /* 32 */  , 0x3eaf32a0U, 0x5bc88e18U, 0xb5673b0aU, 0xd00087b2U, 0x6938502fU, 0x0c5fec97U, 0xe2f05985U, 0x8797e53dU
      /* 40 */  , 0xd1878665U, 0xb4e03addU, 0x5a4f8fcfU, 0x3f283377U, 0x8610e4eaU, 0xe3775852U, 0x0dd8ed40U, 0x68bf51f8U
      /* 48 */  , 0xa1f82bf0U, 0xc49f9748U, 0x2a30225aU, 0x4f579ee2U, 0xf66f497fU, 0x9308f5c7U, 0x7da740d5U, 0x18c0fc6dU
      /* 56 */  , 0x4ed09f35U, 0x2bb7238dU, 0xc518969fU, 0xa07f2a27U, 0x1947fdbaU, 0x7c204102U, 0x928ff410U, 0xf7e848a8U
      /* 64 */  , 0x3d58149bU, 0x583fa823U, 0xb6901d31U, 0xd3f7a189U, 0x6acf7614U, 0x0fa8caacU, 0xe1077fbeU, 0x8460c306U
      /* 72 */  , 0xd270a05eU, 0xb7171ce6U, 0x59b8a9f4U, 0x3cdf154cU, 0x85e7c2d1U, 0xe0807e69U, 0x0e2fcb7bU, 0x6b4877c3U
      /* 80 */  , 0xa20f0dcbU, 0xc768b173U, 0x29c70461U, 0x4ca0b8d9U, 0xf5986f44U, 0x90ffd3fcU, 0x7e5066eeU, 0x1b37da56U
      /* 88 */  , 0x4d27b90eU, 0x284005b6U, 0xc6efb0a4U, 0xa3880c1cU, 0x1ab0db81U, 0x7fd76739U, 0x9178d22bU, 0xf41f6e93U
      /* 96 */  , 0x03f7263bU, 0x66909a83U, 0x883f2f91U, 0xed589329U, 0x546044b4U, 0x3107f80cU, 0xdfa84d1eU, 0xbacff1a6U
      /* 104 */  , 0xecdf92feU, 0x89b82e46U, 0x67179b54U, 0x027027ecU, 0xbb48f071U, 0xde2f4cc9U, 0x3080f9dbU, 0x55e74563U
      /* 112 */  , 0x9ca03f6bU, 0xf9c783d3U, 0x176836c1U, 0x720f8a79U, 0xcb375de4U, 0xae50e15cU, 0x40ff544eU, 0x2598e8f6U
      /* 120 */  , 0x73888baeU, 0x16ef3716U, 0xf8408204U, 0x9d273ebcU, 0x241fe921U, 0x41785599U, 0xafd7e08bU, 0xcab05c33U
      /* 128 */  , 0x3bb659edU, 0x5ed1e555U, 0xb07e5047U, 0xd519ecffU, 0x6c213b62U, 0x094687daU, 0xe7e932c8U, 0x828e8e70U
      /* 136 */  , 0xd49eed28U, 0xb1f95190U, 0x5f56e482U, 0x3a31583aU, 0x83098fa7U, 0xe66e331fU, 0x08c1860dU, 0x6da63ab5U
      /* 144 */  , 0xa4e140bdU, 0xc186fc05U, 0x2f294917U, 0x4a4ef5afU, 0xf3762232U, 0x96119e8aU, 0x78be2b98U, 0x1dd99720U
      /* 152 */  , 0x4bc9f478U, 0x2eae48c0U, 0xc001fdd2U, 0xa566416aU, 0x1c5e96f7U, 0x79392a4fU, 0x97969f5dU, 0xf2f123e5U
      /* 160 */  , 0x05196b4dU, 0x607ed7f5U, 0x8ed162e7U, 0xebb6de5fU, 0x528e09c2U, 0x37e9b57aU, 0xd9460068U, 0xbc21bcd0U
      /* 168 */  , 0xea31df88U, 0x8f566330U, 0x61f9d622U, 0x049e6a9aU, 0xbda6bd07U, 0xd8c101bfU, 0x366eb4adU, 0x53090815U
      /* 176 */  , 0x9a4e721dU, 0xff29cea5U, 0x11867bb7U, 0x74e1c70fU, 0xcdd91092U, 0xa8beac2aU, 0x46111938U, 0x2376a580U
      /* 184 */  , 0x7566c6d8U, 0x10017a60U, 0xfeaecf72U, 0x9bc973caU, 0x22f1a457U, 0x479618efU, 0xa939adfdU, 0xcc5e1145U
      /* 192 */  , 0x06ee4d76U, 0x6389f1ceU, 0x8d2644dcU, 0xe841f864U, 0x51792ff9U, 0x341e9341U, 0xdab12653U, 0xbfd69aebU
      /* 200 */  , 0xe9c6f9b3U, 0x8ca1450bU, 0x620ef019U, 0x07694ca1U, 0xbe519b3cU, 0xdb362784U, 0x35999296U, 0x50fe2e2eU
      /* 208 */  , 0x99b95426U, 0xfcdee89eU, 0x12715d8cU, 0x7716e134U, 0xce2e36a9U, 0xab498a11U, 0x45e63f03U, 0x208183bbU
      /* 216 */  , 0x7691e0e3U, 0x13f65c5bU, 0xfd59e949U, 0x983e55f1U, 0x2106826cU, 0x44613ed4U, 0xaace8bc6U, 0xcfa9377eU
      /* 224 */  , 0x38417fd6U, 0x5d26c36eU, 0xb389767cU, 0xd6eecac4U, 0x6fd61d59U, 0x0ab1a1e1U, 0xe41e14f3U, 0x8179a84bU
      /* 232 */  , 0xd769cb13U, 0xb20e77abU, 0x5ca1c2b9U, 0x39c67e01U, 0x80fea99cU, 0xe5991524U, 0x0b36a036U, 0x6e511c8eU
      /* 240 */  , 0xa7166686U, 0xc271da3eU, 0x2cde6f2cU, 0x49b9d394U, 0xf0810409U, 0x95e6b8b1U, 0x7b490da3U, 0x1e2eb11bU
      /* 248 */  , 0x483ed243U, 0x2d596efbU, 0xc3f6dbe9U, 0xa6916751U, 0x1fa9b0ccU, 0x7ace0c74U, 0x9461b966U, 0xf10605deU
    }
  #endif
  };

juint StubRoutines::zarch::_crc32c_table[CRC32_TABLES][CRC32_COLUMN_SIZE] = {
    /* polyBits = 4394350321 0x0000000105ec76f1L, shifted = 0x82f63b78 */
    /* CRC32C table for single bytes, auto-generated.  DO NOT MODIFY!  */
    /* CRC32C table 0 for quad-bytes (little-endian), auto-generated. DO NOT MODIFY! */
    {
      /* 0 */    0x00000000U, 0xf26b8303U, 0xe13b70f7U, 0x1350f3f4U, 0xc79a971fU, 0x35f1141cU, 0x26a1e7e8U, 0xd4ca64ebU
      /* 8 */  , 0x8ad958cfU, 0x78b2dbccU, 0x6be22838U, 0x9989ab3bU, 0x4d43cfd0U, 0xbf284cd3U, 0xac78bf27U, 0x5e133c24U
      /* 16 */  , 0x105ec76fU, 0xe235446cU, 0xf165b798U, 0x030e349bU, 0xd7c45070U, 0x25afd373U, 0x36ff2087U, 0xc494a384U
      /* 24 */  , 0x9a879fa0U, 0x68ec1ca3U, 0x7bbcef57U, 0x89d76c54U, 0x5d1d08bfU, 0xaf768bbcU, 0xbc267848U, 0x4e4dfb4bU
      /* 32 */  , 0x20bd8edeU, 0xd2d60dddU, 0xc186fe29U, 0x33ed7d2aU, 0xe72719c1U, 0x154c9ac2U, 0x061c6936U, 0xf477ea35U
      /* 40 */  , 0xaa64d611U, 0x580f5512U, 0x4b5fa6e6U, 0xb93425e5U, 0x6dfe410eU, 0x9f95c20dU, 0x8cc531f9U, 0x7eaeb2faU
      /* 48 */  , 0x30e349b1U, 0xc288cab2U, 0xd1d83946U, 0x23b3ba45U, 0xf779deaeU, 0x05125dadU, 0x1642ae59U, 0xe4292d5aU
      /* 56 */  , 0xba3a117eU, 0x4851927dU, 0x5b016189U, 0xa96ae28aU, 0x7da08661U, 0x8fcb0562U, 0x9c9bf696U, 0x6ef07595U
      /* 64 */  , 0x417b1dbcU, 0xb3109ebfU, 0xa0406d4bU, 0x522bee48U, 0x86e18aa3U, 0x748a09a0U, 0x67dafa54U, 0x95b17957U
      /* 72 */  , 0xcba24573U, 0x39c9c670U, 0x2a993584U, 0xd8f2b687U, 0x0c38d26cU, 0xfe53516fU, 0xed03a29bU, 0x1f682198U
      /* 80 */  , 0x5125dad3U, 0xa34e59d0U, 0xb01eaa24U, 0x42752927U, 0x96bf4dccU, 0x64d4cecfU, 0x77843d3bU, 0x85efbe38U
      /* 88 */  , 0xdbfc821cU, 0x2997011fU, 0x3ac7f2ebU, 0xc8ac71e8U, 0x1c661503U, 0xee0d9600U, 0xfd5d65f4U, 0x0f36e6f7U
      /* 96 */  , 0x61c69362U, 0x93ad1061U, 0x80fde395U, 0x72966096U, 0xa65c047dU, 0x5437877eU, 0x4767748aU, 0xb50cf789U
      /* 104 */  , 0xeb1fcbadU, 0x197448aeU, 0x0a24bb5aU, 0xf84f3859U, 0x2c855cb2U, 0xdeeedfb1U, 0xcdbe2c45U, 0x3fd5af46U
      /* 112 */  , 0x7198540dU, 0x83f3d70eU, 0x90a324faU, 0x62c8a7f9U, 0xb602c312U, 0x44694011U, 0x5739b3e5U, 0xa55230e6U
      /* 120 */  , 0xfb410cc2U, 0x092a8fc1U, 0x1a7a7c35U, 0xe811ff36U, 0x3cdb9bddU, 0xceb018deU, 0xdde0eb2aU, 0x2f8b6829U
      /* 128 */  , 0x82f63b78U, 0x709db87bU, 0x63cd4b8fU, 0x91a6c88cU, 0x456cac67U, 0xb7072f64U, 0xa457dc90U, 0x563c5f93U
      /* 136 */  , 0x082f63b7U, 0xfa44e0b4U, 0xe9141340U, 0x1b7f9043U, 0xcfb5f4a8U, 0x3dde77abU, 0x2e8e845fU, 0xdce5075cU
      /* 144 */  , 0x92a8fc17U, 0x60c37f14U, 0x73938ce0U, 0x81f80fe3U, 0x55326b08U, 0xa759e80bU, 0xb4091bffU, 0x466298fcU
      /* 152 */  , 0x1871a4d8U, 0xea1a27dbU, 0xf94ad42fU, 0x0b21572cU, 0xdfeb33c7U, 0x2d80b0c4U, 0x3ed04330U, 0xccbbc033U
      /* 160 */  , 0xa24bb5a6U, 0x502036a5U, 0x4370c551U, 0xb11b4652U, 0x65d122b9U, 0x97baa1baU, 0x84ea524eU, 0x7681d14dU
      /* 168 */  , 0x2892ed69U, 0xdaf96e6aU, 0xc9a99d9eU, 0x3bc21e9dU, 0xef087a76U, 0x1d63f975U, 0x0e330a81U, 0xfc588982U
      /* 176 */  , 0xb21572c9U, 0x407ef1caU, 0x532e023eU, 0xa145813dU, 0x758fe5d6U, 0x87e466d5U, 0x94b49521U, 0x66df1622U
      /* 184 */  , 0x38cc2a06U, 0xcaa7a905U, 0xd9f75af1U, 0x2b9cd9f2U, 0xff56bd19U, 0x0d3d3e1aU, 0x1e6dcdeeU, 0xec064eedU
      /* 192 */  , 0xc38d26c4U, 0x31e6a5c7U, 0x22b65633U, 0xd0ddd530U, 0x0417b1dbU, 0xf67c32d8U, 0xe52cc12cU, 0x1747422fU
      /* 200 */  , 0x49547e0bU, 0xbb3ffd08U, 0xa86f0efcU, 0x5a048dffU, 0x8ecee914U, 0x7ca56a17U, 0x6ff599e3U, 0x9d9e1ae0U
      /* 208 */  , 0xd3d3e1abU, 0x21b862a8U, 0x32e8915cU, 0xc083125fU, 0x144976b4U, 0xe622f5b7U, 0xf5720643U, 0x07198540U
      /* 216 */  , 0x590ab964U, 0xab613a67U, 0xb831c993U, 0x4a5a4a90U, 0x9e902e7bU, 0x6cfbad78U, 0x7fab5e8cU, 0x8dc0dd8fU
      /* 224 */  , 0xe330a81aU, 0x115b2b19U, 0x020bd8edU, 0xf0605beeU, 0x24aa3f05U, 0xd6c1bc06U, 0xc5914ff2U, 0x37faccf1U
      /* 232 */  , 0x69e9f0d5U, 0x9b8273d6U, 0x88d28022U, 0x7ab90321U, 0xae7367caU, 0x5c18e4c9U, 0x4f48173dU, 0xbd23943eU
      /* 240 */  , 0xf36e6f75U, 0x0105ec76U, 0x12551f82U, 0xe03e9c81U, 0x34f4f86aU, 0xc69f7b69U, 0xd5cf889dU, 0x27a40b9eU
      /* 248 */  , 0x79b737baU, 0x8bdcb4b9U, 0x988c474dU, 0x6ae7c44eU, 0xbe2da0a5U, 0x4c4623a6U, 0x5f16d052U, 0xad7d5351U
    }
  #ifdef  CRC32_BYFOUR
    ,
    /* CRC32C table 1 for quad-bytes (little-endian), auto-generated. DO NOT MODIFY! */
    {
      /* 0 */    0x00000000U, 0x13a29877U, 0x274530eeU, 0x34e7a899U, 0x4e8a61dcU, 0x5d28f9abU, 0x69cf5132U, 0x7a6dc945U
      /* 8 */  , 0x9d14c3b8U, 0x8eb65bcfU, 0xba51f356U, 0xa9f36b21U, 0xd39ea264U, 0xc03c3a13U, 0xf4db928aU, 0xe7790afdU
      /* 16 */  , 0x3fc5f181U, 0x2c6769f6U, 0x1880c16fU, 0x0b225918U, 0x714f905dU, 0x62ed082aU, 0x560aa0b3U, 0x45a838c4U
      /* 24 */  , 0xa2d13239U, 0xb173aa4eU, 0x859402d7U, 0x96369aa0U, 0xec5b53e5U, 0xfff9cb92U, 0xcb1e630bU, 0xd8bcfb7cU
      /* 32 */  , 0x7f8be302U, 0x6c297b75U, 0x58ced3ecU, 0x4b6c4b9bU, 0x310182deU, 0x22a31aa9U, 0x1644b230U, 0x05e62a47U
      /* 40 */  , 0xe29f20baU, 0xf13db8cdU, 0xc5da1054U, 0xd6788823U, 0xac154166U, 0xbfb7d911U, 0x8b507188U, 0x98f2e9ffU
      /* 48 */  , 0x404e1283U, 0x53ec8af4U, 0x670b226dU, 0x74a9ba1aU, 0x0ec4735fU, 0x1d66eb28U, 0x298143b1U, 0x3a23dbc6U
      /* 56 */  , 0xdd5ad13bU, 0xcef8494cU, 0xfa1fe1d5U, 0xe9bd79a2U, 0x93d0b0e7U, 0x80722890U, 0xb4958009U, 0xa737187eU
      /* 64 */  , 0xff17c604U, 0xecb55e73U, 0xd852f6eaU, 0xcbf06e9dU, 0xb19da7d8U, 0xa23f3fafU, 0x96d89736U, 0x857a0f41U
      /* 72 */  , 0x620305bcU, 0x71a19dcbU, 0x45463552U, 0x56e4ad25U, 0x2c896460U, 0x3f2bfc17U, 0x0bcc548eU, 0x186eccf9U
      /* 80 */  , 0xc0d23785U, 0xd370aff2U, 0xe797076bU, 0xf4359f1cU, 0x8e585659U, 0x9dface2eU, 0xa91d66b7U, 0xbabffec0U
      /* 88 */  , 0x5dc6f43dU, 0x4e646c4aU, 0x7a83c4d3U, 0x69215ca4U, 0x134c95e1U, 0x00ee0d96U, 0x3409a50fU, 0x27ab3d78U
      /* 96 */  , 0x809c2506U, 0x933ebd71U, 0xa7d915e8U, 0xb47b8d9fU, 0xce1644daU, 0xddb4dcadU, 0xe9537434U, 0xfaf1ec43U
      /* 104 */  , 0x1d88e6beU, 0x0e2a7ec9U, 0x3acdd650U, 0x296f4e27U, 0x53028762U, 0x40a01f15U, 0x7447b78cU, 0x67e52ffbU
      /* 112 */  , 0xbf59d487U, 0xacfb4cf0U, 0x981ce469U, 0x8bbe7c1eU, 0xf1d3b55bU, 0xe2712d2cU, 0xd69685b5U, 0xc5341dc2U
      /* 120 */  , 0x224d173fU, 0x31ef8f48U, 0x050827d1U, 0x16aabfa6U, 0x6cc776e3U, 0x7f65ee94U, 0x4b82460dU, 0x5820de7aU
      /* 128 */  , 0xfbc3faf9U, 0xe861628eU, 0xdc86ca17U, 0xcf245260U, 0xb5499b25U, 0xa6eb0352U, 0x920cabcbU, 0x81ae33bcU
      /* 136 */  , 0x66d73941U, 0x7575a136U, 0x419209afU, 0x523091d8U, 0x285d589dU, 0x3bffc0eaU, 0x0f186873U, 0x1cbaf004U
      /* 144 */  , 0xc4060b78U, 0xd7a4930fU, 0xe3433b96U, 0xf0e1a3e1U, 0x8a8c6aa4U, 0x992ef2d3U, 0xadc95a4aU, 0xbe6bc23dU
      /* 152 */  , 0x5912c8c0U, 0x4ab050b7U, 0x7e57f82eU, 0x6df56059U, 0x1798a91cU, 0x043a316bU, 0x30dd99f2U, 0x237f0185U
      /* 160 */  , 0x844819fbU, 0x97ea818cU, 0xa30d2915U, 0xb0afb162U, 0xcac27827U, 0xd960e050U, 0xed8748c9U, 0xfe25d0beU
      /* 168 */  , 0x195cda43U, 0x0afe4234U, 0x3e19eaadU, 0x2dbb72daU, 0x57d6bb9fU, 0x447423e8U, 0x70938b71U, 0x63311306U
      /* 176 */  , 0xbb8de87aU, 0xa82f700dU, 0x9cc8d894U, 0x8f6a40e3U, 0xf50789a6U, 0xe6a511d1U, 0xd242b948U, 0xc1e0213fU
      /* 184 */  , 0x26992bc2U, 0x353bb3b5U, 0x01dc1b2cU, 0x127e835bU, 0x68134a1eU, 0x7bb1d269U, 0x4f567af0U, 0x5cf4e287U
      /* 192 */  , 0x04d43cfdU, 0x1776a48aU, 0x23910c13U, 0x30339464U, 0x4a5e5d21U, 0x59fcc556U, 0x6d1b6dcfU, 0x7eb9f5b8U
      /* 200 */  , 0x99c0ff45U, 0x8a626732U, 0xbe85cfabU, 0xad2757dcU, 0xd74a9e99U, 0xc4e806eeU, 0xf00fae77U, 0xe3ad3600U
      /* 208 */  , 0x3b11cd7cU, 0x28b3550bU, 0x1c54fd92U, 0x0ff665e5U, 0x759baca0U, 0x663934d7U, 0x52de9c4eU, 0x417c0439U
      /* 216 */  , 0xa6050ec4U, 0xb5a796b3U, 0x81403e2aU, 0x92e2a65dU, 0xe88f6f18U, 0xfb2df76fU, 0xcfca5ff6U, 0xdc68c781U
      /* 224 */  , 0x7b5fdfffU, 0x68fd4788U, 0x5c1aef11U, 0x4fb87766U, 0x35d5be23U, 0x26772654U, 0x12908ecdU, 0x013216baU
      /* 232 */  , 0xe64b1c47U, 0xf5e98430U, 0xc10e2ca9U, 0xd2acb4deU, 0xa8c17d9bU, 0xbb63e5ecU, 0x8f844d75U, 0x9c26d502U
      /* 240 */  , 0x449a2e7eU, 0x5738b609U, 0x63df1e90U, 0x707d86e7U, 0x0a104fa2U, 0x19b2d7d5U, 0x2d557f4cU, 0x3ef7e73bU
      /* 248 */  , 0xd98eedc6U, 0xca2c75b1U, 0xfecbdd28U, 0xed69455fU, 0x97048c1aU, 0x84a6146dU, 0xb041bcf4U, 0xa3e32483U
    }
    ,
    /* CRC32C table 2 for quad-bytes (little-endian), auto-generated. DO NOT MODIFY! */
    {
      /* 0 */    0x00000000U, 0xa541927eU, 0x4f6f520dU, 0xea2ec073U, 0x9edea41aU, 0x3b9f3664U, 0xd1b1f617U, 0x74f06469U
      /* 8 */  , 0x38513ec5U, 0x9d10acbbU, 0x773e6cc8U, 0xd27ffeb6U, 0xa68f9adfU, 0x03ce08a1U, 0xe9e0c8d2U, 0x4ca15aacU
      /* 16 */  , 0x70a27d8aU, 0xd5e3eff4U, 0x3fcd2f87U, 0x9a8cbdf9U, 0xee7cd990U, 0x4b3d4beeU, 0xa1138b9dU, 0x045219e3U
      /* 24 */  , 0x48f3434fU, 0xedb2d131U, 0x079c1142U, 0xa2dd833cU, 0xd62de755U, 0x736c752bU, 0x9942b558U, 0x3c032726U
      /* 32 */  , 0xe144fb14U, 0x4405696aU, 0xae2ba919U, 0x0b6a3b67U, 0x7f9a5f0eU, 0xdadbcd70U, 0x30f50d03U, 0x95b49f7dU
      /* 40 */  , 0xd915c5d1U, 0x7c5457afU, 0x967a97dcU, 0x333b05a2U, 0x47cb61cbU, 0xe28af3b5U, 0x08a433c6U, 0xade5a1b8U
      /* 48 */  , 0x91e6869eU, 0x34a714e0U, 0xde89d493U, 0x7bc846edU, 0x0f382284U, 0xaa79b0faU, 0x40577089U, 0xe516e2f7U
      /* 56 */  , 0xa9b7b85bU, 0x0cf62a25U, 0xe6d8ea56U, 0x43997828U, 0x37691c41U, 0x92288e3fU, 0x78064e4cU, 0xdd47dc32U
      /* 64 */  , 0xc76580d9U, 0x622412a7U, 0x880ad2d4U, 0x2d4b40aaU, 0x59bb24c3U, 0xfcfab6bdU, 0x16d476ceU, 0xb395e4b0U
      /* 72 */  , 0xff34be1cU, 0x5a752c62U, 0xb05bec11U, 0x151a7e6fU, 0x61ea1a06U, 0xc4ab8878U, 0x2e85480bU, 0x8bc4da75U
      /* 80 */  , 0xb7c7fd53U, 0x12866f2dU, 0xf8a8af5eU, 0x5de93d20U, 0x29195949U, 0x8c58cb37U, 0x66760b44U, 0xc337993aU
      /* 88 */  , 0x8f96c396U, 0x2ad751e8U, 0xc0f9919bU, 0x65b803e5U, 0x1148678cU, 0xb409f5f2U, 0x5e273581U, 0xfb66a7ffU
      /* 96 */  , 0x26217bcdU, 0x8360e9b3U, 0x694e29c0U, 0xcc0fbbbeU, 0xb8ffdfd7U, 0x1dbe4da9U, 0xf7908ddaU, 0x52d11fa4U
      /* 104 */  , 0x1e704508U, 0xbb31d776U, 0x511f1705U, 0xf45e857bU, 0x80aee112U, 0x25ef736cU, 0xcfc1b31fU, 0x6a802161U
      /* 112 */  , 0x56830647U, 0xf3c29439U, 0x19ec544aU, 0xbcadc634U, 0xc85da25dU, 0x6d1c3023U, 0x8732f050U, 0x2273622eU
      /* 120 */  , 0x6ed23882U, 0xcb93aafcU, 0x21bd6a8fU, 0x84fcf8f1U, 0xf00c9c98U, 0x554d0ee6U, 0xbf63ce95U, 0x1a225cebU
      /* 128 */  , 0x8b277743U, 0x2e66e53dU, 0xc448254eU, 0x6109b730U, 0x15f9d359U, 0xb0b84127U, 0x5a968154U, 0xffd7132aU
      /* 136 */  , 0xb3764986U, 0x1637dbf8U, 0xfc191b8bU, 0x595889f5U, 0x2da8ed9cU, 0x88e97fe2U, 0x62c7bf91U, 0xc7862defU
      /* 144 */  , 0xfb850ac9U, 0x5ec498b7U, 0xb4ea58c4U, 0x11abcabaU, 0x655baed3U, 0xc01a3cadU, 0x2a34fcdeU, 0x8f756ea0U
      /* 152 */  , 0xc3d4340cU, 0x6695a672U, 0x8cbb6601U, 0x29faf47fU, 0x5d0a9016U, 0xf84b0268U, 0x1265c21bU, 0xb7245065U
      /* 160 */  , 0x6a638c57U, 0xcf221e29U, 0x250cde5aU, 0x804d4c24U, 0xf4bd284dU, 0x51fcba33U, 0xbbd27a40U, 0x1e93e83eU
      /* 168 */  , 0x5232b292U, 0xf77320ecU, 0x1d5de09fU, 0xb81c72e1U, 0xccec1688U, 0x69ad84f6U, 0x83834485U, 0x26c2d6fbU
      /* 176 */  , 0x1ac1f1ddU, 0xbf8063a3U, 0x55aea3d0U, 0xf0ef31aeU, 0x841f55c7U, 0x215ec7b9U, 0xcb7007caU, 0x6e3195b4U
      /* 184 */  , 0x2290cf18U, 0x87d15d66U, 0x6dff9d15U, 0xc8be0f6bU, 0xbc4e6b02U, 0x190ff97cU, 0xf321390fU, 0x5660ab71U
      /* 192 */  , 0x4c42f79aU, 0xe90365e4U, 0x032da597U, 0xa66c37e9U, 0xd29c5380U, 0x77ddc1feU, 0x9df3018dU, 0x38b293f3U
      /* 200 */  , 0x7413c95fU, 0xd1525b21U, 0x3b7c9b52U, 0x9e3d092cU, 0xeacd6d45U, 0x4f8cff3bU, 0xa5a23f48U, 0x00e3ad36U
      /* 208 */  , 0x3ce08a10U, 0x99a1186eU, 0x738fd81dU, 0xd6ce4a63U, 0xa23e2e0aU, 0x077fbc74U, 0xed517c07U, 0x4810ee79U
      /* 216 */  , 0x04b1b4d5U, 0xa1f026abU, 0x4bdee6d8U, 0xee9f74a6U, 0x9a6f10cfU, 0x3f2e82b1U, 0xd50042c2U, 0x7041d0bcU
      /* 224 */  , 0xad060c8eU, 0x08479ef0U, 0xe2695e83U, 0x4728ccfdU, 0x33d8a894U, 0x96993aeaU, 0x7cb7fa99U, 0xd9f668e7U
      /* 232 */  , 0x9557324bU, 0x3016a035U, 0xda386046U, 0x7f79f238U, 0x0b899651U, 0xaec8042fU, 0x44e6c45cU, 0xe1a75622U
      /* 240 */  , 0xdda47104U, 0x78e5e37aU, 0x92cb2309U, 0x378ab177U, 0x437ad51eU, 0xe63b4760U, 0x0c158713U, 0xa954156dU
      /* 248 */  , 0xe5f54fc1U, 0x40b4ddbfU, 0xaa9a1dccU, 0x0fdb8fb2U, 0x7b2bebdbU, 0xde6a79a5U, 0x3444b9d6U, 0x91052ba8U
    }
    ,
    /* CRC32C table 3 for quad-bytes (little-endian), auto-generated. DO NOT MODIFY! */
    {
      /* 0 */    0x00000000U, 0xdd45aab8U, 0xbf672381U, 0x62228939U, 0x7b2231f3U, 0xa6679b4bU, 0xc4451272U, 0x1900b8caU
      /* 8 */  , 0xf64463e6U, 0x2b01c95eU, 0x49234067U, 0x9466eadfU, 0x8d665215U, 0x5023f8adU, 0x32017194U, 0xef44db2cU
      /* 16 */  , 0xe964b13dU, 0x34211b85U, 0x560392bcU, 0x8b463804U, 0x924680ceU, 0x4f032a76U, 0x2d21a34fU, 0xf06409f7U
      /* 24 */  , 0x1f20d2dbU, 0xc2657863U, 0xa047f15aU, 0x7d025be2U, 0x6402e328U, 0xb9474990U, 0xdb65c0a9U, 0x06206a11U
      /* 32 */  , 0xd725148bU, 0x0a60be33U, 0x6842370aU, 0xb5079db2U, 0xac072578U, 0x71428fc0U, 0x136006f9U, 0xce25ac41U
      /* 40 */  , 0x2161776dU, 0xfc24ddd5U, 0x9e0654ecU, 0x4343fe54U, 0x5a43469eU, 0x8706ec26U, 0xe524651fU, 0x3861cfa7U
      /* 48 */  , 0x3e41a5b6U, 0xe3040f0eU, 0x81268637U, 0x5c632c8fU, 0x45639445U, 0x98263efdU, 0xfa04b7c4U, 0x27411d7cU
      /* 56 */  , 0xc805c650U, 0x15406ce8U, 0x7762e5d1U, 0xaa274f69U, 0xb327f7a3U, 0x6e625d1bU, 0x0c40d422U, 0xd1057e9aU
      /* 64 */  , 0xaba65fe7U, 0x76e3f55fU, 0x14c17c66U, 0xc984d6deU, 0xd0846e14U, 0x0dc1c4acU, 0x6fe34d95U, 0xb2a6e72dU
      /* 72 */  , 0x5de23c01U, 0x80a796b9U, 0xe2851f80U, 0x3fc0b538U, 0x26c00df2U, 0xfb85a74aU, 0x99a72e73U, 0x44e284cbU
      /* 80 */  , 0x42c2eedaU, 0x9f874462U, 0xfda5cd5bU, 0x20e067e3U, 0x39e0df29U, 0xe4a57591U, 0x8687fca8U, 0x5bc25610U
      /* 88 */  , 0xb4868d3cU, 0x69c32784U, 0x0be1aebdU, 0xd6a40405U, 0xcfa4bccfU, 0x12e11677U, 0x70c39f4eU, 0xad8635f6U
      /* 96 */  , 0x7c834b6cU, 0xa1c6e1d4U, 0xc3e468edU, 0x1ea1c255U, 0x07a17a9fU, 0xdae4d027U, 0xb8c6591eU, 0x6583f3a6U
      /* 104 */  , 0x8ac7288aU, 0x57828232U, 0x35a00b0bU, 0xe8e5a1b3U, 0xf1e51979U, 0x2ca0b3c1U, 0x4e823af8U, 0x93c79040U
      /* 112 */  , 0x95e7fa51U, 0x48a250e9U, 0x2a80d9d0U, 0xf7c57368U, 0xeec5cba2U, 0x3380611aU, 0x51a2e823U, 0x8ce7429bU
      /* 120 */  , 0x63a399b7U, 0xbee6330fU, 0xdcc4ba36U, 0x0181108eU, 0x1881a844U, 0xc5c402fcU, 0xa7e68bc5U, 0x7aa3217dU
      /* 128 */  , 0x52a0c93fU, 0x8fe56387U, 0xedc7eabeU, 0x30824006U, 0x2982f8ccU, 0xf4c75274U, 0x96e5db4dU, 0x4ba071f5U
      /* 136 */  , 0xa4e4aad9U, 0x79a10061U, 0x1b838958U, 0xc6c623e0U, 0xdfc69b2aU, 0x02833192U, 0x60a1b8abU, 0xbde41213U
      /* 144 */  , 0xbbc47802U, 0x6681d2baU, 0x04a35b83U, 0xd9e6f13bU, 0xc0e649f1U, 0x1da3e349U, 0x7f816a70U, 0xa2c4c0c8U
      /* 152 */  , 0x4d801be4U, 0x90c5b15cU, 0xf2e73865U, 0x2fa292ddU, 0x36a22a17U, 0xebe780afU, 0x89c50996U, 0x5480a32eU
      /* 160 */  , 0x8585ddb4U, 0x58c0770cU, 0x3ae2fe35U, 0xe7a7548dU, 0xfea7ec47U, 0x23e246ffU, 0x41c0cfc6U, 0x9c85657eU
      /* 168 */  , 0x73c1be52U, 0xae8414eaU, 0xcca69dd3U, 0x11e3376bU, 0x08e38fa1U, 0xd5a62519U, 0xb784ac20U, 0x6ac10698U
      /* 176 */  , 0x6ce16c89U, 0xb1a4c631U, 0xd3864f08U, 0x0ec3e5b0U, 0x17c35d7aU, 0xca86f7c2U, 0xa8a47efbU, 0x75e1d443U
      /* 184 */  , 0x9aa50f6fU, 0x47e0a5d7U, 0x25c22ceeU, 0xf8878656U, 0xe1873e9cU, 0x3cc29424U, 0x5ee01d1dU, 0x83a5b7a5U
      /* 192 */  , 0xf90696d8U, 0x24433c60U, 0x4661b559U, 0x9b241fe1U, 0x8224a72bU, 0x5f610d93U, 0x3d4384aaU, 0xe0062e12U
      /* 200 */  , 0x0f42f53eU, 0xd2075f86U, 0xb025d6bfU, 0x6d607c07U, 0x7460c4cdU, 0xa9256e75U, 0xcb07e74cU, 0x16424df4U
      /* 208 */  , 0x106227e5U, 0xcd278d5dU, 0xaf050464U, 0x7240aedcU, 0x6b401616U, 0xb605bcaeU, 0xd4273597U, 0x09629f2fU
      /* 216 */  , 0xe6264403U, 0x3b63eebbU, 0x59416782U, 0x8404cd3aU, 0x9d0475f0U, 0x4041df48U, 0x22635671U, 0xff26fcc9U
      /* 224 */  , 0x2e238253U, 0xf36628ebU, 0x9144a1d2U, 0x4c010b6aU, 0x5501b3a0U, 0x88441918U, 0xea669021U, 0x37233a99U
      /* 232 */  , 0xd867e1b5U, 0x05224b0dU, 0x6700c234U, 0xba45688cU, 0xa345d046U, 0x7e007afeU, 0x1c22f3c7U, 0xc167597fU
      /* 240 */  , 0xc747336eU, 0x1a0299d6U, 0x782010efU, 0xa565ba57U, 0xbc65029dU, 0x6120a825U, 0x0302211cU, 0xde478ba4U
      /* 248 */  , 0x31035088U, 0xec46fa30U, 0x8e647309U, 0x5321d9b1U, 0x4a21617bU, 0x9764cbc3U, 0xf54642faU, 0x2803e842U
    }
    ,
    /* CRC32C table 4 for quad-bytes (  big-endian ), auto-generated. DO NOT MODIFY! */
    {
      /* 0 */    0x00000000U, 0x03836bf2U, 0xf7703be1U, 0xf4f35013U, 0x1f979ac7U, 0x1c14f135U, 0xe8e7a126U, 0xeb64cad4U
      /* 8 */  , 0xcf58d98aU, 0xccdbb278U, 0x3828e26bU, 0x3bab8999U, 0xd0cf434dU, 0xd34c28bfU, 0x27bf78acU, 0x243c135eU
      /* 16 */  , 0x6fc75e10U, 0x6c4435e2U, 0x98b765f1U, 0x9b340e03U, 0x7050c4d7U, 0x73d3af25U, 0x8720ff36U, 0x84a394c4U
      /* 24 */  , 0xa09f879aU, 0xa31cec68U, 0x57efbc7bU, 0x546cd789U, 0xbf081d5dU, 0xbc8b76afU, 0x487826bcU, 0x4bfb4d4eU
      /* 32 */  , 0xde8ebd20U, 0xdd0dd6d2U, 0x29fe86c1U, 0x2a7ded33U, 0xc11927e7U, 0xc29a4c15U, 0x36691c06U, 0x35ea77f4U
      /* 40 */  , 0x11d664aaU, 0x12550f58U, 0xe6a65f4bU, 0xe52534b9U, 0x0e41fe6dU, 0x0dc2959fU, 0xf931c58cU, 0xfab2ae7eU
      /* 48 */  , 0xb149e330U, 0xb2ca88c2U, 0x4639d8d1U, 0x45bab323U, 0xaede79f7U, 0xad5d1205U, 0x59ae4216U, 0x5a2d29e4U
      /* 56 */  , 0x7e113abaU, 0x7d925148U, 0x8961015bU, 0x8ae26aa9U, 0x6186a07dU, 0x6205cb8fU, 0x96f69b9cU, 0x9575f06eU
      /* 64 */  , 0xbc1d7b41U, 0xbf9e10b3U, 0x4b6d40a0U, 0x48ee2b52U, 0xa38ae186U, 0xa0098a74U, 0x54fada67U, 0x5779b195U
      /* 72 */  , 0x7345a2cbU, 0x70c6c939U, 0x8435992aU, 0x87b6f2d8U, 0x6cd2380cU, 0x6f5153feU, 0x9ba203edU, 0x9821681fU
      /* 80 */  , 0xd3da2551U, 0xd0594ea3U, 0x24aa1eb0U, 0x27297542U, 0xcc4dbf96U, 0xcfced464U, 0x3b3d8477U, 0x38beef85U
      /* 88 */  , 0x1c82fcdbU, 0x1f019729U, 0xebf2c73aU, 0xe871acc8U, 0x0315661cU, 0x00960deeU, 0xf4655dfdU, 0xf7e6360fU
      /* 96 */  , 0x6293c661U, 0x6110ad93U, 0x95e3fd80U, 0x96609672U, 0x7d045ca6U, 0x7e873754U, 0x8a746747U, 0x89f70cb5U
      /* 104 */  , 0xadcb1febU, 0xae487419U, 0x5abb240aU, 0x59384ff8U, 0xb25c852cU, 0xb1dfeedeU, 0x452cbecdU, 0x46afd53fU
      /* 112 */  , 0x0d549871U, 0x0ed7f383U, 0xfa24a390U, 0xf9a7c862U, 0x12c302b6U, 0x11406944U, 0xe5b33957U, 0xe63052a5U
      /* 120 */  , 0xc20c41fbU, 0xc18f2a09U, 0x357c7a1aU, 0x36ff11e8U, 0xdd9bdb3cU, 0xde18b0ceU, 0x2aebe0ddU, 0x29688b2fU
      /* 128 */  , 0x783bf682U, 0x7bb89d70U, 0x8f4bcd63U, 0x8cc8a691U, 0x67ac6c45U, 0x642f07b7U, 0x90dc57a4U, 0x935f3c56U
      /* 136 */  , 0xb7632f08U, 0xb4e044faU, 0x401314e9U, 0x43907f1bU, 0xa8f4b5cfU, 0xab77de3dU, 0x5f848e2eU, 0x5c07e5dcU
      /* 144 */  , 0x17fca892U, 0x147fc360U, 0xe08c9373U, 0xe30ff881U, 0x086b3255U, 0x0be859a7U, 0xff1b09b4U, 0xfc986246U
      /* 152 */  , 0xd8a47118U, 0xdb271aeaU, 0x2fd44af9U, 0x2c57210bU, 0xc733ebdfU, 0xc4b0802dU, 0x3043d03eU, 0x33c0bbccU
      /* 160 */  , 0xa6b54ba2U, 0xa5362050U, 0x51c57043U, 0x52461bb1U, 0xb922d165U, 0xbaa1ba97U, 0x4e52ea84U, 0x4dd18176U
      /* 168 */  , 0x69ed9228U, 0x6a6ef9daU, 0x9e9da9c9U, 0x9d1ec23bU, 0x767a08efU, 0x75f9631dU, 0x810a330eU, 0x828958fcU
      /* 176 */  , 0xc97215b2U, 0xcaf17e40U, 0x3e022e53U, 0x3d8145a1U, 0xd6e58f75U, 0xd566e487U, 0x2195b494U, 0x2216df66U
      /* 184 */  , 0x062acc38U, 0x05a9a7caU, 0xf15af7d9U, 0xf2d99c2bU, 0x19bd56ffU, 0x1a3e3d0dU, 0xeecd6d1eU, 0xed4e06ecU
      /* 192 */  , 0xc4268dc3U, 0xc7a5e631U, 0x3356b622U, 0x30d5ddd0U, 0xdbb11704U, 0xd8327cf6U, 0x2cc12ce5U, 0x2f424717U
      /* 200 */  , 0x0b7e5449U, 0x08fd3fbbU, 0xfc0e6fa8U, 0xff8d045aU, 0x14e9ce8eU, 0x176aa57cU, 0xe399f56fU, 0xe01a9e9dU
      /* 208 */  , 0xabe1d3d3U, 0xa862b821U, 0x5c91e832U, 0x5f1283c0U, 0xb4764914U, 0xb7f522e6U, 0x430672f5U, 0x40851907U
      /* 216 */  , 0x64b90a59U, 0x673a61abU, 0x93c931b8U, 0x904a5a4aU, 0x7b2e909eU, 0x78adfb6cU, 0x8c5eab7fU, 0x8fddc08dU
      /* 224 */  , 0x1aa830e3U, 0x192b5b11U, 0xedd80b02U, 0xee5b60f0U, 0x053faa24U, 0x06bcc1d6U, 0xf24f91c5U, 0xf1ccfa37U
      /* 232 */  , 0xd5f0e969U, 0xd673829bU, 0x2280d288U, 0x2103b97aU, 0xca6773aeU, 0xc9e4185cU, 0x3d17484fU, 0x3e9423bdU
      /* 240 */  , 0x756f6ef3U, 0x76ec0501U, 0x821f5512U, 0x819c3ee0U, 0x6af8f434U, 0x697b9fc6U, 0x9d88cfd5U, 0x9e0ba427U
      /* 248 */  , 0xba37b779U, 0xb9b4dc8bU, 0x4d478c98U, 0x4ec4e76aU, 0xa5a02dbeU, 0xa623464cU, 0x52d0165fU, 0x51537dadU
    }
    ,
    /* CRC32C table 5 for quad-bytes (  big-endian ), auto-generated. DO NOT MODIFY! */
    {
      /* 0 */    0x00000000U, 0x7798a213U, 0xee304527U, 0x99a8e734U, 0xdc618a4eU, 0xabf9285dU, 0x3251cf69U, 0x45c96d7aU
      /* 8 */  , 0xb8c3149dU, 0xcf5bb68eU, 0x56f351baU, 0x216bf3a9U, 0x64a29ed3U, 0x133a3cc0U, 0x8a92dbf4U, 0xfd0a79e7U
      /* 16 */  , 0x81f1c53fU, 0xf669672cU, 0x6fc18018U, 0x1859220bU, 0x5d904f71U, 0x2a08ed62U, 0xb3a00a56U, 0xc438a845U
      /* 24 */  , 0x3932d1a2U, 0x4eaa73b1U, 0xd7029485U, 0xa09a3696U, 0xe5535becU, 0x92cbf9ffU, 0x0b631ecbU, 0x7cfbbcd8U
      /* 32 */  , 0x02e38b7fU, 0x757b296cU, 0xecd3ce58U, 0x9b4b6c4bU, 0xde820131U, 0xa91aa322U, 0x30b24416U, 0x472ae605U
      /* 40 */  , 0xba209fe2U, 0xcdb83df1U, 0x5410dac5U, 0x238878d6U, 0x664115acU, 0x11d9b7bfU, 0x8871508bU, 0xffe9f298U
      /* 48 */  , 0x83124e40U, 0xf48aec53U, 0x6d220b67U, 0x1abaa974U, 0x5f73c40eU, 0x28eb661dU, 0xb1438129U, 0xc6db233aU
      /* 56 */  , 0x3bd15addU, 0x4c49f8ceU, 0xd5e11ffaU, 0xa279bde9U, 0xe7b0d093U, 0x90287280U, 0x098095b4U, 0x7e1837a7U
      /* 64 */  , 0x04c617ffU, 0x735eb5ecU, 0xeaf652d8U, 0x9d6ef0cbU, 0xd8a79db1U, 0xaf3f3fa2U, 0x3697d896U, 0x410f7a85U
      /* 72 */  , 0xbc050362U, 0xcb9da171U, 0x52354645U, 0x25ade456U, 0x6064892cU, 0x17fc2b3fU, 0x8e54cc0bU, 0xf9cc6e18U
      /* 80 */  , 0x8537d2c0U, 0xf2af70d3U, 0x6b0797e7U, 0x1c9f35f4U, 0x5956588eU, 0x2ecefa9dU, 0xb7661da9U, 0xc0febfbaU
      /* 88 */  , 0x3df4c65dU, 0x4a6c644eU, 0xd3c4837aU, 0xa45c2169U, 0xe1954c13U, 0x960dee00U, 0x0fa50934U, 0x783dab27U
      /* 96 */  , 0x06259c80U, 0x71bd3e93U, 0xe815d9a7U, 0x9f8d7bb4U, 0xda4416ceU, 0xaddcb4ddU, 0x347453e9U, 0x43ecf1faU
      /* 104 */  , 0xbee6881dU, 0xc97e2a0eU, 0x50d6cd3aU, 0x274e6f29U, 0x62870253U, 0x151fa040U, 0x8cb74774U, 0xfb2fe567U
      /* 112 */  , 0x87d459bfU, 0xf04cfbacU, 0x69e41c98U, 0x1e7cbe8bU, 0x5bb5d3f1U, 0x2c2d71e2U, 0xb58596d6U, 0xc21d34c5U
      /* 120 */  , 0x3f174d22U, 0x488fef31U, 0xd1270805U, 0xa6bfaa16U, 0xe376c76cU, 0x94ee657fU, 0x0d46824bU, 0x7ade2058U
      /* 128 */  , 0xf9fac3fbU, 0x8e6261e8U, 0x17ca86dcU, 0x605224cfU, 0x259b49b5U, 0x5203eba6U, 0xcbab0c92U, 0xbc33ae81U
      /* 136 */  , 0x4139d766U, 0x36a17575U, 0xaf099241U, 0xd8913052U, 0x9d585d28U, 0xeac0ff3bU, 0x7368180fU, 0x04f0ba1cU
      /* 144 */  , 0x780b06c4U, 0x0f93a4d7U, 0x963b43e3U, 0xe1a3e1f0U, 0xa46a8c8aU, 0xd3f22e99U, 0x4a5ac9adU, 0x3dc26bbeU
      /* 152 */  , 0xc0c81259U, 0xb750b04aU, 0x2ef8577eU, 0x5960f56dU, 0x1ca99817U, 0x6b313a04U, 0xf299dd30U, 0x85017f23U
      /* 160 */  , 0xfb194884U, 0x8c81ea97U, 0x15290da3U, 0x62b1afb0U, 0x2778c2caU, 0x50e060d9U, 0xc94887edU, 0xbed025feU
      /* 168 */  , 0x43da5c19U, 0x3442fe0aU, 0xadea193eU, 0xda72bb2dU, 0x9fbbd657U, 0xe8237444U, 0x718b9370U, 0x06133163U
      /* 176 */  , 0x7ae88dbbU, 0x0d702fa8U, 0x94d8c89cU, 0xe3406a8fU, 0xa68907f5U, 0xd111a5e6U, 0x48b942d2U, 0x3f21e0c1U
      /* 184 */  , 0xc22b9926U, 0xb5b33b35U, 0x2c1bdc01U, 0x5b837e12U, 0x1e4a1368U, 0x69d2b17bU, 0xf07a564fU, 0x87e2f45cU
      /* 192 */  , 0xfd3cd404U, 0x8aa47617U, 0x130c9123U, 0x64943330U, 0x215d5e4aU, 0x56c5fc59U, 0xcf6d1b6dU, 0xb8f5b97eU
      /* 200 */  , 0x45ffc099U, 0x3267628aU, 0xabcf85beU, 0xdc5727adU, 0x999e4ad7U, 0xee06e8c4U, 0x77ae0ff0U, 0x0036ade3U
      /* 208 */  , 0x7ccd113bU, 0x0b55b328U, 0x92fd541cU, 0xe565f60fU, 0xa0ac9b75U, 0xd7343966U, 0x4e9cde52U, 0x39047c41U
      /* 216 */  , 0xc40e05a6U, 0xb396a7b5U, 0x2a3e4081U, 0x5da6e292U, 0x186f8fe8U, 0x6ff72dfbU, 0xf65fcacfU, 0x81c768dcU
      /* 224 */  , 0xffdf5f7bU, 0x8847fd68U, 0x11ef1a5cU, 0x6677b84fU, 0x23bed535U, 0x54267726U, 0xcd8e9012U, 0xba163201U
      /* 232 */  , 0x471c4be6U, 0x3084e9f5U, 0xa92c0ec1U, 0xdeb4acd2U, 0x9b7dc1a8U, 0xece563bbU, 0x754d848fU, 0x02d5269cU
      /* 240 */  , 0x7e2e9a44U, 0x09b63857U, 0x901edf63U, 0xe7867d70U, 0xa24f100aU, 0xd5d7b219U, 0x4c7f552dU, 0x3be7f73eU
      /* 248 */  , 0xc6ed8ed9U, 0xb1752ccaU, 0x28ddcbfeU, 0x5f4569edU, 0x1a8c0497U, 0x6d14a684U, 0xf4bc41b0U, 0x8324e3a3U
    }
    ,
    /* CRC32C table 6 for quad-bytes (  big-endian ), auto-generated. DO NOT MODIFY! */
    {
      /* 0 */    0x00000000U, 0x7e9241a5U, 0x0d526f4fU, 0x73c02eeaU, 0x1aa4de9eU, 0x64369f3bU, 0x17f6b1d1U, 0x6964f074U
      /* 8 */  , 0xc53e5138U, 0xbbac109dU, 0xc86c3e77U, 0xb6fe7fd2U, 0xdf9a8fa6U, 0xa108ce03U, 0xd2c8e0e9U, 0xac5aa14cU
      /* 16 */  , 0x8a7da270U, 0xf4efe3d5U, 0x872fcd3fU, 0xf9bd8c9aU, 0x90d97ceeU, 0xee4b3d4bU, 0x9d8b13a1U, 0xe3195204U
      /* 24 */  , 0x4f43f348U, 0x31d1b2edU, 0x42119c07U, 0x3c83dda2U, 0x55e72dd6U, 0x2b756c73U, 0x58b54299U, 0x2627033cU
      /* 32 */  , 0x14fb44e1U, 0x6a690544U, 0x19a92baeU, 0x673b6a0bU, 0x0e5f9a7fU, 0x70cddbdaU, 0x030df530U, 0x7d9fb495U
      /* 40 */  , 0xd1c515d9U, 0xaf57547cU, 0xdc977a96U, 0xa2053b33U, 0xcb61cb47U, 0xb5f38ae2U, 0xc633a408U, 0xb8a1e5adU
      /* 48 */  , 0x9e86e691U, 0xe014a734U, 0x93d489deU, 0xed46c87bU, 0x8422380fU, 0xfab079aaU, 0x89705740U, 0xf7e216e5U
      /* 56 */  , 0x5bb8b7a9U, 0x252af60cU, 0x56ead8e6U, 0x28789943U, 0x411c6937U, 0x3f8e2892U, 0x4c4e0678U, 0x32dc47ddU
      /* 64 */  , 0xd98065c7U, 0xa7122462U, 0xd4d20a88U, 0xaa404b2dU, 0xc324bb59U, 0xbdb6fafcU, 0xce76d416U, 0xb0e495b3U
      /* 72 */  , 0x1cbe34ffU, 0x622c755aU, 0x11ec5bb0U, 0x6f7e1a15U, 0x061aea61U, 0x7888abc4U, 0x0b48852eU, 0x75dac48bU
      /* 80 */  , 0x53fdc7b7U, 0x2d6f8612U, 0x5eafa8f8U, 0x203de95dU, 0x49591929U, 0x37cb588cU, 0x440b7666U, 0x3a9937c3U
      /* 88 */  , 0x96c3968fU, 0xe851d72aU, 0x9b91f9c0U, 0xe503b865U, 0x8c674811U, 0xf2f509b4U, 0x8135275eU, 0xffa766fbU
      /* 96 */  , 0xcd7b2126U, 0xb3e96083U, 0xc0294e69U, 0xbebb0fccU, 0xd7dfffb8U, 0xa94dbe1dU, 0xda8d90f7U, 0xa41fd152U
      /* 104 */  , 0x0845701eU, 0x76d731bbU, 0x05171f51U, 0x7b855ef4U, 0x12e1ae80U, 0x6c73ef25U, 0x1fb3c1cfU, 0x6121806aU
      /* 112 */  , 0x47068356U, 0x3994c2f3U, 0x4a54ec19U, 0x34c6adbcU, 0x5da25dc8U, 0x23301c6dU, 0x50f03287U, 0x2e627322U
      /* 120 */  , 0x8238d26eU, 0xfcaa93cbU, 0x8f6abd21U, 0xf1f8fc84U, 0x989c0cf0U, 0xe60e4d55U, 0x95ce63bfU, 0xeb5c221aU
      /* 128 */  , 0x4377278bU, 0x3de5662eU, 0x4e2548c4U, 0x30b70961U, 0x59d3f915U, 0x2741b8b0U, 0x5481965aU, 0x2a13d7ffU
      /* 136 */  , 0x864976b3U, 0xf8db3716U, 0x8b1b19fcU, 0xf5895859U, 0x9ceda82dU, 0xe27fe988U, 0x91bfc762U, 0xef2d86c7U
      /* 144 */  , 0xc90a85fbU, 0xb798c45eU, 0xc458eab4U, 0xbacaab11U, 0xd3ae5b65U, 0xad3c1ac0U, 0xdefc342aU, 0xa06e758fU
      /* 152 */  , 0x0c34d4c3U, 0x72a69566U, 0x0166bb8cU, 0x7ff4fa29U, 0x16900a5dU, 0x68024bf8U, 0x1bc26512U, 0x655024b7U
      /* 160 */  , 0x578c636aU, 0x291e22cfU, 0x5ade0c25U, 0x244c4d80U, 0x4d28bdf4U, 0x33bafc51U, 0x407ad2bbU, 0x3ee8931eU
      /* 168 */  , 0x92b23252U, 0xec2073f7U, 0x9fe05d1dU, 0xe1721cb8U, 0x8816ecccU, 0xf684ad69U, 0x85448383U, 0xfbd6c226U
      /* 176 */  , 0xddf1c11aU, 0xa36380bfU, 0xd0a3ae55U, 0xae31eff0U, 0xc7551f84U, 0xb9c75e21U, 0xca0770cbU, 0xb495316eU
      /* 184 */  , 0x18cf9022U, 0x665dd187U, 0x159dff6dU, 0x6b0fbec8U, 0x026b4ebcU, 0x7cf90f19U, 0x0f3921f3U, 0x71ab6056U
      /* 192 */  , 0x9af7424cU, 0xe46503e9U, 0x97a52d03U, 0xe9376ca6U, 0x80539cd2U, 0xfec1dd77U, 0x8d01f39dU, 0xf393b238U
      /* 200 */  , 0x5fc91374U, 0x215b52d1U, 0x529b7c3bU, 0x2c093d9eU, 0x456dcdeaU, 0x3bff8c4fU, 0x483fa2a5U, 0x36ade300U
      /* 208 */  , 0x108ae03cU, 0x6e18a199U, 0x1dd88f73U, 0x634aced6U, 0x0a2e3ea2U, 0x74bc7f07U, 0x077c51edU, 0x79ee1048U
      /* 216 */  , 0xd5b4b104U, 0xab26f0a1U, 0xd8e6de4bU, 0xa6749feeU, 0xcf106f9aU, 0xb1822e3fU, 0xc24200d5U, 0xbcd04170U
      /* 224 */  , 0x8e0c06adU, 0xf09e4708U, 0x835e69e2U, 0xfdcc2847U, 0x94a8d833U, 0xea3a9996U, 0x99fab77cU, 0xe768f6d9U
      /* 232 */  , 0x4b325795U, 0x35a01630U, 0x466038daU, 0x38f2797fU, 0x5196890bU, 0x2f04c8aeU, 0x5cc4e644U, 0x2256a7e1U
      /* 240 */  , 0x0471a4ddU, 0x7ae3e578U, 0x0923cb92U, 0x77b18a37U, 0x1ed57a43U, 0x60473be6U, 0x1387150cU, 0x6d1554a9U
      /* 248 */  , 0xc14ff5e5U, 0xbfddb440U, 0xcc1d9aaaU, 0xb28fdb0fU, 0xdbeb2b7bU, 0xa5796adeU, 0xd6b94434U, 0xa82b0591U
    }
    ,
    /* CRC32C table 7 for quad-bytes (  big-endian ), auto-generated. DO NOT MODIFY! */
    {
      /* 0 */    0x00000000U, 0xb8aa45ddU, 0x812367bfU, 0x39892262U, 0xf331227bU, 0x4b9b67a6U, 0x721245c4U, 0xcab80019U
      /* 8 */  , 0xe66344f6U, 0x5ec9012bU, 0x67402349U, 0xdfea6694U, 0x1552668dU, 0xadf82350U, 0x94710132U, 0x2cdb44efU
      /* 16 */  , 0x3db164e9U, 0x851b2134U, 0xbc920356U, 0x0438468bU, 0xce804692U, 0x762a034fU, 0x4fa3212dU, 0xf70964f0U
      /* 24 */  , 0xdbd2201fU, 0x637865c2U, 0x5af147a0U, 0xe25b027dU, 0x28e30264U, 0x904947b9U, 0xa9c065dbU, 0x116a2006U
      /* 32 */  , 0x8b1425d7U, 0x33be600aU, 0x0a374268U, 0xb29d07b5U, 0x782507acU, 0xc08f4271U, 0xf9066013U, 0x41ac25ceU
      /* 40 */  , 0x6d776121U, 0xd5dd24fcU, 0xec54069eU, 0x54fe4343U, 0x9e46435aU, 0x26ec0687U, 0x1f6524e5U, 0xa7cf6138U
      /* 48 */  , 0xb6a5413eU, 0x0e0f04e3U, 0x37862681U, 0x8f2c635cU, 0x45946345U, 0xfd3e2698U, 0xc4b704faU, 0x7c1d4127U
      /* 56 */  , 0x50c605c8U, 0xe86c4015U, 0xd1e56277U, 0x694f27aaU, 0xa3f727b3U, 0x1b5d626eU, 0x22d4400cU, 0x9a7e05d1U
      /* 64 */  , 0xe75fa6abU, 0x5ff5e376U, 0x667cc114U, 0xded684c9U, 0x146e84d0U, 0xacc4c10dU, 0x954de36fU, 0x2de7a6b2U
      /* 72 */  , 0x013ce25dU, 0xb996a780U, 0x801f85e2U, 0x38b5c03fU, 0xf20dc026U, 0x4aa785fbU, 0x732ea799U, 0xcb84e244U
      /* 80 */  , 0xdaeec242U, 0x6244879fU, 0x5bcda5fdU, 0xe367e020U, 0x29dfe039U, 0x9175a5e4U, 0xa8fc8786U, 0x1056c25bU
      /* 88 */  , 0x3c8d86b4U, 0x8427c369U, 0xbdaee10bU, 0x0504a4d6U, 0xcfbca4cfU, 0x7716e112U, 0x4e9fc370U, 0xf63586adU
      /* 96 */  , 0x6c4b837cU, 0xd4e1c6a1U, 0xed68e4c3U, 0x55c2a11eU, 0x9f7aa107U, 0x27d0e4daU, 0x1e59c6b8U, 0xa6f38365U
      /* 104 */  , 0x8a28c78aU, 0x32828257U, 0x0b0ba035U, 0xb3a1e5e8U, 0x7919e5f1U, 0xc1b3a02cU, 0xf83a824eU, 0x4090c793U
      /* 112 */  , 0x51fae795U, 0xe950a248U, 0xd0d9802aU, 0x6873c5f7U, 0xa2cbc5eeU, 0x1a618033U, 0x23e8a251U, 0x9b42e78cU
      /* 120 */  , 0xb799a363U, 0x0f33e6beU, 0x36bac4dcU, 0x8e108101U, 0x44a88118U, 0xfc02c4c5U, 0xc58be6a7U, 0x7d21a37aU
      /* 128 */  , 0x3fc9a052U, 0x8763e58fU, 0xbeeac7edU, 0x06408230U, 0xccf88229U, 0x7452c7f4U, 0x4ddbe596U, 0xf571a04bU
      /* 136 */  , 0xd9aae4a4U, 0x6100a179U, 0x5889831bU, 0xe023c6c6U, 0x2a9bc6dfU, 0x92318302U, 0xabb8a160U, 0x1312e4bdU
      /* 144 */  , 0x0278c4bbU, 0xbad28166U, 0x835ba304U, 0x3bf1e6d9U, 0xf149e6c0U, 0x49e3a31dU, 0x706a817fU, 0xc8c0c4a2U
      /* 152 */  , 0xe41b804dU, 0x5cb1c590U, 0x6538e7f2U, 0xdd92a22fU, 0x172aa236U, 0xaf80e7ebU, 0x9609c589U, 0x2ea38054U
      /* 160 */  , 0xb4dd8585U, 0x0c77c058U, 0x35fee23aU, 0x8d54a7e7U, 0x47eca7feU, 0xff46e223U, 0xc6cfc041U, 0x7e65859cU
      /* 168 */  , 0x52bec173U, 0xea1484aeU, 0xd39da6ccU, 0x6b37e311U, 0xa18fe308U, 0x1925a6d5U, 0x20ac84b7U, 0x9806c16aU
      /* 176 */  , 0x896ce16cU, 0x31c6a4b1U, 0x084f86d3U, 0xb0e5c30eU, 0x7a5dc317U, 0xc2f786caU, 0xfb7ea4a8U, 0x43d4e175U
      /* 184 */  , 0x6f0fa59aU, 0xd7a5e047U, 0xee2cc225U, 0x568687f8U, 0x9c3e87e1U, 0x2494c23cU, 0x1d1de05eU, 0xa5b7a583U
      /* 192 */  , 0xd89606f9U, 0x603c4324U, 0x59b56146U, 0xe11f249bU, 0x2ba72482U, 0x930d615fU, 0xaa84433dU, 0x122e06e0U
      /* 200 */  , 0x3ef5420fU, 0x865f07d2U, 0xbfd625b0U, 0x077c606dU, 0xcdc46074U, 0x756e25a9U, 0x4ce707cbU, 0xf44d4216U
      /* 208 */  , 0xe5276210U, 0x5d8d27cdU, 0x640405afU, 0xdcae4072U, 0x1616406bU, 0xaebc05b6U, 0x973527d4U, 0x2f9f6209U
      /* 216 */  , 0x034426e6U, 0xbbee633bU, 0x82674159U, 0x3acd0484U, 0xf075049dU, 0x48df4140U, 0x71566322U, 0xc9fc26ffU
      /* 224 */  , 0x5382232eU, 0xeb2866f3U, 0xd2a14491U, 0x6a0b014cU, 0xa0b30155U, 0x18194488U, 0x219066eaU, 0x993a2337U
      /* 232 */  , 0xb5e167d8U, 0x0d4b2205U, 0x34c20067U, 0x8c6845baU, 0x46d045a3U, 0xfe7a007eU, 0xc7f3221cU, 0x7f5967c1U
      /* 240 */  , 0x6e3347c7U, 0xd699021aU, 0xef102078U, 0x57ba65a5U, 0x9d0265bcU, 0x25a82061U, 0x1c210203U, 0xa48b47deU
      /* 248 */  , 0x88500331U, 0x30fa46ecU, 0x0973648eU, 0xb1d92153U, 0x7b61214aU, 0xc3cb6497U, 0xfa4246f5U, 0x42e80328U
    }
  #endif
  };
