// MOLDYN.CPP

// Copyright (C) 1998 Tommi Hassinen.

// This package is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This package is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this package; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

/*################################################################################################*/

#include "moldyn.h"

#include "atom.h"
#include "model.h"

#include <sstream>
using namespace std;

#define T_BUFF_SIZE	500

/*################################################################################################*/

moldyn::moldyn(engine * p1, f64 p2)
{
	eng = p1;
	
	tstep1 = p2;						// [1.0e-15 s]
	tstep2 = tstep1 * tstep1;				// [1.0e-30 s^2]
	
	vel = new f64[eng->GetAtomCount() * 3];			// [1.0e+3 m/s]
	acc = new f64[eng->GetAtomCount() * 3];			// [1.0e+12 m/s^2]
	
	mass = new f64[eng->GetAtomCount()];			// [kg/mol]
	
	locked = new char[eng->GetAtomCount()];
	
	step_counter = 0;
	
	sum_of_masses = 0.0;					// [kg/mol]
	
	atom ** glob_atmtab = eng->GetSetup()->GetAtoms();
	
	num_locked = 0; i32s counter = 0;
	while (counter < eng->GetAtomCount())
	{
		bool lflag = false;
		if (glob_atmtab[counter]->flags & ATOMFLAG_USER_LOCKED)
		{
			lflag = true;
			num_locked++;
		}
		
		mass[counter] = glob_atmtab[counter]->mass;
		mass[counter] *= 1.6605402e-27 * 6.0221367e+23;
		
		sum_of_masses += mass[counter];		// kg/mol ; all atoms.
		
		locked[counter] = lflag;
		
		for (i32s n1 = 0;n1 < 3;n1++)
		{
			vel[counter * 3 + n1] = 0.0;
			acc[counter * 3 + n1] = 0.0;
		}
		
		counter++;
	}
	
	// the rest are just default values; can be modified...
	// the rest are just default values; can be modified...
	// the rest are just default values; can be modified...
	
	target_temperature = 300.0;	// [K]
	temperature_rtime = 100.0;	// [fs] ; should be > 100x timestep.
	
	target_pressure = 1.000;	// [bar]
	pressure_rtime = 1000.0;	// [fs] ; should be > 100x timestep.
	isoth_compr = 4.57e-5;		// [1/bar]
}

moldyn::~moldyn(void)
{
	delete[] vel;
	delete[] acc;
	
	delete[] mass;
	
	delete[] locked;
}

void moldyn::TakeMDStep(bool enable_temperature_control, bool enable_pressure_control)
{
	for (i32s n1 = 0;n1 < eng->GetAtomCount();n1++)
	{
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			f64 tmpA = acc[n1 * 3 + n2];
			
			f64 tmp1 = tstep1 * vel[n1 * 3 + n2] * 1.0e-3;
			f64 tmp2 = tstep2 * tmpA * 0.5e-9;
			
			if (!locked[n1])
			{
				eng->crd[n1 * 3 + n2] += tmp1 + tmp2;
				
				vel[n1 * 3 + n2] += tstep1 * tmpA * 0.5e-6;
			}
		}
	}
	
	const bool report_shake = !(step_counter % 1000);
	eng->DoSHAKE(report_shake);
	
	eng->Compute(1, enable_pressure_control);	// ask to calculate virial if pressure needed...
	epot = eng->energy;
	
	for (i32s n1 = 0;n1 < eng->GetAtomCount();n1++)
	{
		if (locked[n1])
		{
			acc[n1 * 3 + 0] = acc[n1 * 3 + 1] = acc[n1 * 3 + 2] = 0.0;
			vel[n1 * 3 + 0] = vel[n1 * 3 + 1] = vel[n1 * 3 + 2] = 0.0;
		}
		else
		{
			acc[n1 * 3 + 0] = -eng->d1[n1 * 3 + 0] / mass[n1];
			acc[n1 * 3 + 1] = -eng->d1[n1 * 3 + 1] / mass[n1];
			acc[n1 * 3 + 2] = -eng->d1[n1 * 3 + 2] / mass[n1];
			
			vel[n1 * 3 + 0] += tstep1 * acc[n1 * 3 + 0] * 0.5e-6;
			vel[n1 * 3 + 1] += tstep1 * acc[n1 * 3 + 1] * 0.5e-6;
			vel[n1 * 3 + 2] += tstep1 * acc[n1 * 3 + 2] * 0.5e-6;
		}
	}
	
	f64 ekinCOMP[3];
	ekin = KineticEnergy(ekinCOMP);
	
	f64 current_temperature = ConvEKinTemp(ekin);
	
	if (enable_temperature_control)
	{
		// do a Berendsen-type temperature control step.
		// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
		
		const f64 tmp1 = (target_temperature / ConvEKinTemp(ekin)) - 1.0;
		const f64 tmp2 = tstep1 / temperature_rtime;
		const f64 delta = sqrt(1.0 + tmp2 * tmp1);
		
		ekin *= delta;
		ekinCOMP[0] *= delta;
		ekinCOMP[1] *= delta;
		ekinCOMP[2] *= delta;
		
		SetEKin(ekin);
		current_temperature = ConvEKinTemp(ekin);
	}
	
	// the pressure computation is based on:
	// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
	// Haile JM : "##Molecular dynamics simulation / Elementary methods"
	// the GROMACS manual (v3) ; www.gromacs.org / documentation / manuals
	
	if (enable_pressure_control)
	{
		// the unit of virial is this:
		// [nm] * [kJ / (mol * nm)] = [kJ/mol] ; the kilo must be handled.
		
		// volume must be m^3 / mol (multiplied by Avogadro constant).
		
		f64 pressure[3] =
		{
			target_pressure,
			target_pressure,
			target_pressure
		};
		
		f64 volume;
		
	// eng_pbc != NULL if we will use a system with periodic boundary conditions...
		engine_pbc * eng_pbc = dynamic_cast<engine_pbc *>(eng);
		if (eng_pbc != NULL)
		{
			f64 tmpVOL = eng_pbc->box_HALFdim[0] * eng_pbc->box_HALFdim[1] * eng_pbc->box_HALFdim[2];
			
			// exponent = 23 - 9 - 9 - 9 = -4
			// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
			// -9 comes from nanometers (in all tree dimensions).
			
			tmpVOL *= 8.0 * 6.0221367e-4;	// 2^3 = 8 for half-dimensions.
			volume = tmpVOL;		// store it as m^3 / mol...
			
			// +3 comes from the kilo prefix of Ekin and virial.
			// -5 comes from Pa -> bar conversion.
			
			for (i32s n1 = 0;n1 < 3;n1++)
			{
				pressure[n1] = 1.0e-2 * (2.0 * ekinCOMP[n1] + eng->virial[n1]) / tmpVOL;
			}
		}
		
		// the pressure components are now in bar units.
		// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
		
		const f64 tot_P = (pressure[0] + pressure[1] + pressure[2]) / 3.0;
		
		// do a Berendsen-type pressure control step.
		// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
		
		f64 tmp1 = (target_pressure - tot_P);
		if (tmp1 < -100.0) tmp1 = -100.0;	// damp too large steps...
		if (tmp1 > +100.0) tmp1 = +100.0;	// damp too large steps...
		
		const f64 tmp2 = isoth_compr * tstep1 / pressure_rtime;
		const f64 delta = pow(1.0 - tmp2 * tmp1, 0.3333333333333333);
		
		eng->ScaleCRD(delta, delta, delta);
		
		if (eng_pbc != NULL)
		{
			eng_pbc->box_HALFdim[0] *= delta;
			eng_pbc->box_HALFdim[1] *= delta;
			eng_pbc->box_HALFdim[2] *= delta;
			
			model * mdl = eng->GetSetup()->GetModel();
			mdl->saved_periodic_box_HALFdim[0] = eng_pbc->box_HALFdim[0];
			mdl->saved_periodic_box_HALFdim[1] = eng_pbc->box_HALFdim[1];
			mdl->saved_periodic_box_HALFdim[2] = eng_pbc->box_HALFdim[2];
		}
		
		saved_pressure = tot_P;
		saved_density = 0.001 * sum_of_masses / volume;		// kg/dm^3
	}
	
	step_counter++;
}

f64 moldyn::KineticEnergy(f64 * comp)
{
	if (comp != NULL)
	{
		comp[0] = 0.0;
		comp[1] = 0.0;
		comp[2] = 0.0;
	}
	
	f64 energy = 0.0;
	
	for (i32s n1 = 0;n1 < eng->GetAtomCount();n1++)
	{
		if (locked[n1]) continue;
		
		const f64 tmpX = 500.0 * mass[n1];
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			const f64 tmp1 = vel[n1 * 3 + n2];
			const f64 tmp2 = tmpX * tmp1 * tmp1;
			
			energy += tmp2;
			
			if (comp != NULL)
			{
				comp[n2] += tmp2;
			}
		}
	}
	
	return energy;
}

// <Ekin> = 3/2 * k * T		// <Ekin> = <0.5 * m * v^2) = average kinetic energy of a molecule.
// EKin = 3N/2 * k * T		// Ekin = total kinetic energy, N = number of atoms (3N = degrees of freedom).
// EKin = 3N/2 * R * T		// if Ekin is expressed "per mole", change the constant k to per-mole-constant R.

f64 moldyn::ConvTempEKin(f64 p1)
{
	return (3.0 / 2.0) * p1 * ((eng->GetAtomCount() - num_locked) * 8.314510) / 1000.0;
}

f64 moldyn::ConvEKinTemp(f64 p1)
{
	return (2.0 / 3.0) * p1 * 1000.0 / ((eng->GetAtomCount() - num_locked) * 8.314510);
}

void moldyn::SetEKin(f64 p1)
{
	f64 tmp1 = p1 / KineticEnergy();
	f64 tmp2 = (tmp1 < 0.0 ? 0.0 : sqrt(tmp1));
	
	for (i32s n1 = 0;n1 < eng->GetAtomCount();n1++)
	{
		if (locked[n1])
		{
			vel[n1 * 3 + 0] = 0.0;
			vel[n1 * 3 + 1] = 0.0;
			vel[n1 * 3 + 2] = 0.0;
		}
		else
		{
			vel[n1 * 3 + 0] *= tmp2;
			vel[n1 * 3 + 1] *= tmp2;
			vel[n1 * 3 + 2] *= tmp2;
		}
	}
}

/*################################################################################################*/

// eof
