package tests::ReportTest;

use strict;

use base qw/Lire::Test::TestCase/;

use Lire::Report;
use Lire::Report::Section;
use Lire::Report::Subreport;
use File::Temp qw/ tempfile /;
use POSIX qw/strftime/;

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->set_up_tz( 'UTC' );

    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    return;
}

sub test_new {
    my $self = $_[0];


    my $report = new Lire::Report();
    $self->assert_isa( 'Lire::Report', $report );
    $self->assert_null( $report->{'_period'} );

    my $now = time;
    $report = new Lire::Report( 'daily', $now, $now + 3600 );
    $self->assert_str_equals( 'daily', $report->{'_period'} );
    $self->assert_num_equals( $now, $report->{'_start'} );
    $self->assert_num_equals( $now + 3600, $report->{'_end'} );

}

sub test_timespan_start {
    my $self = $_[0];

    my $report = new Lire::Report();
    $report->timespan_start( 10_000 );
    $self->assert_num_equals( 10_000, $report->{'_start'} );
    $self->assert_num_equals( 10_000, $report->timespan_start() );

    $report->{'_end'} = 90_000;
    $self->assert_dies( qr/'start' parameter is greater than timespan_end\(\)/,
                        sub { $report->timespan_start( 100_000 ) } );
    $report->timespan_start( '' );
    $self->assert_null( $report->timespan_start() );
    $report->timespan_start( undef );
    $self->assert_null( $report->timespan_start() );
    $report->timespan_start( 0 );
    $self->assert_null( $report->timespan_start() );
}

sub test_timespan_end {
    my $self = $_[0];

    my $report = new Lire::Report();
    $report->timespan_end( 10_000 );
    $self->assert_num_equals( 10_000, $report->{'_end'} );
    $self->assert_num_equals( 10_000, $report->timespan_end() );

    $report->{'_start'} = 90_000;
    $self->assert_dies( qr/'end' parameter is smaller than timespan_start\(\)/,
                        sub { $report->timespan_end( 80_000 ) } );
    $report->timespan_end( '' );
    $self->assert_null( $report->timespan_end() );
    $report->timespan_end( undef );
    $self->assert_null( $report->timespan_end() );
    $report->timespan_end( 0 );
    $self->assert_null( $report->timespan_end() );
}

my $empty_report = <<EOX;
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE lire:report PUBLIC
  "-//LogReport.ORG//DTD Lire Report Markup Language V2.1//EN"
  "http://www.logreport.org/LRML/2.1/lrml.dtd">
<lire:report xmlns:lire="http://www.logreport.org/LRML/" xmlns:lrcml="http://www.logreport.org/LRCML/" version="2.1">
 <!-- generated by Lire::Report(3pm) -->
 <lire:title>Title with an eacute: \xc3\xa9</lire:title>
 <lire:date time="&NOW&">&NOW_FMT&</lire:date>
 <lire:timespan period="daily" start="" end="">Unknown Period</lire:timespan>
</lire:report>
EOX

sub test_write_report_empty {
    my $self = $_[0];

    my ( $fh, $file ) = tempfile();
    my $warn = "";
    local $SIG{'__WARN__'} = sub { $warn .= join "", @_ };
    my $report = new Lire::Report( "daily" );
    $report->title( "Title with an eacute: \x{00e9}" );
    my $now = $report->date();
    my $now_string = strftime( '%Y-%m-%d %H:%M:%S %Z', localtime( $now ));
    $empty_report =~ s/&NOW&/$now/;
    $empty_report =~ s/&NOW_FMT&/$now_string/;

    $report->write_report( $fh );
    $self->annotate( $warn );
    $self->assert( !$warn, "There were warnings" );

    close $fh;
    open $fh, $file
      or $self->error( "open failed: $!" );
    local $/ = undef;
    my $buf = <$fh>;
    $self->assert_str_equals( $empty_report, $buf );
}

sub test_create_subreport_id {
    my $self = $_[0];

    my $report = new Lire::Report();
    $self->assert_str_equals( 'top-dirs.0',
                              $report->create_subreport_id( 'top-dirs' ) );
    $self->assert_str_equals( 'top-dirs.1',
                              $report->create_subreport_id( 'top-dirs' ) );
    $self->assert_str_equals( 'top-files.0',
                              $report->create_subreport_id( 'top-files' ) );
}

sub test_subreport_by_id {
    my $self = $_[0];

    my $report = new Lire::Report;
    my $section1 = new Lire::Report::Section( 'Section 1' );
    my $section2 = new Lire::Report::Section( 'Section 2' );
    my $sub1 = new Lire::Report::Subreport( 'test', 'top-dirs' );
    $sub1->id( 'top-dirs.0' );
    my $sub2 = new Lire::Report::Subreport( 'test', 'top-files' );
    $sub2->id( 'top-files.0' );
    $section1->add_subreport( $sub1 );
    $section1->add_subreport( $sub2 );
    my $sub3 = new Lire::Report::Subreport( 'test', 'top-dirs' );
    $sub3->id( 'top-dirs.1' );
    $section2->add_subreport( $sub3 );
    $report->add_section( $section1 );
    $report->add_section( $section2 );

    $self->assert_null( $report->subreport_by_id( 'unknown' ) );
    $self->assert_deep_equals( $sub1, $report->subreport_by_id( 'top-dirs.0'));
    $self->assert_deep_equals( $sub2, $report->subreport_by_id('top-files.0'));
    $self->assert_deep_equals( $sub3, $report->subreport_by_id( 'top-dirs.1'));
}

sub test_schemas {
    my $self = $_[0];

    my $report = new Lire::Report();
    my $sect1 = new Lire::Report::Section( 'section 1' );
    $report->add_section( $sect1 );

    my $sub = new Lire::Report::Subreport( 'test', 'top-files' );
    $sub->id( 'test.0' );
    $sub->schemas( 'test' );
    $sect1->add_subreport( $sub );

    my $sect2 = new Lire::Report::Section( 'section 2' );
    $sub = new Lire::Report::Subreport( 'test', 'top-dirs' );
    $sub->id( 'test.1' );
    $sub->schemas( 'test', 'test-extended' );
    $sect1->add_subreport( $sub );
    $report->add_section( $sect2 );

    $self->assert_deep_equals( [ 'test', 'test-extended' ],
                               $report->schemas() );
}

sub test_Section_new {
    my $self = $_[0];

    my $section = new Lire::Report::Section( 'My title' );
    $self->assert_isa( 'Lire::Report::Section', $section );
    $self->assert_str_equals( 'My title', $section->{'_title'} );
    $self->assert_str_equals( 'My title', $section->title() );
    $self->assert_str_equals( $section->{'_title'},
                              $section->title( 'New title' ) );
    $self->assert_deep_equals( [], $section->{'_subreports'} );
}

sub test_Section_add_subreport {
    my $self = $_[0];

    my $section = new Lire::Report::Section( 'My title' );
    my $subreport = new Lire::Report::Subreport( 'test', 'top-dirs' );
    $self->assert_dies( qr/can't add a Subreport which doesn't have an id/,
                        sub { $section->add_subreport( $subreport ) } );
    $subreport->id( 'top-dirs.0' );
    $section->add_subreport( $subreport );
    $self->assert_deep_equals( [ $subreport ], [ $section->subreports() ] );
}

1;

