/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <QtCore/QUrl>

#include "listingmodel.h"

#include "listingwidget.h"

ListingWidget::ListingWidget( QWidget *parent )
  : QTextBrowser( parent ), mModel( 0 )
{
}

void ListingWidget::setModel( ListingModel *model )
{
  if ( mModel )
    disconnect( mModel, SIGNAL( changed() ), this, SLOT( update() ) );

  mModel = model;

  connect( mModel, SIGNAL( layoutChanged() ), this, SLOT( update() ) );
}

void ListingWidget::update()
{
  clear();

  if ( !mModel )
    return;

  QString body;

  body += "<html>\n";
  body += "<body>\n";

  QUrl url;
  url.setScheme( "add" );
  body += QString( "<a href=\"%1\">%2</a>\n" )
                 .arg( url.toString() )
                 .arg( tr( "Create New" ) );

  body += "<center>\n";
  body += "<table width=\"100%\" cellpadding=\"0\" cellspacing=\"1px\" bgcolor=\"black\" border=\"0px\">\n";

  // create table header
  body += "<tr>\n";
  for ( int column = 0; column < mModel->columnCount(); ++column )
    body += QString( "<th><font color=\"#FFFFFF\">%1</font></th>\n" ).arg( mModel->headerData( column, Qt::Horizontal ).toString() );
  body += QString( "<th colspan=\"2\"><font color=\"#FFFFFF\">%1</font></th>" ).arg( tr( "Actions" ) );
  body += "</tr>\n";

  // create table body
  for ( int row = 0; row < mModel->rowCount(); ++row ) {
    body += QString( "<tr bgcolor=\"%1\">\n" ).arg( row % 2 == 0 ? "#C0CDE0" : "#D0DDF0" );
    for ( int column = 0; column < mModel->columnCount(); ++column ) {
      const QModelIndex index = mModel->index( row, column );
      body += QString( "<td>%1</td>\n" ).arg( mModel->data( index ).toString() );
    }

    const QModelIndex index = mModel->index( row, 0 );
    if ( mModel->data( index, ListingModel::HasModifyActionRole ).toBool() ) {
      QUrl url;
      url.setScheme( "modify" );
      url.addQueryItem( "id", QString::number( row ) );

      const bool hasDeleteAction = mModel->data( index, ListingModel::HasDeleteActionRole ).toBool();
      const QString colspan = QString( "colspan=\"%1\"" ).arg( hasDeleteAction ? "1" : "2" );

      /**
       * If the model returns an empty id, we don't show an active link.
       */
      if ( mModel->data( index, ListingModel::ModifyActionIdRole ).toString().isEmpty() ) {
        body += QString( "<td align=\"center\" %1>%2</td>\n" )
                       .arg( colspan )
                       .arg( mModel->data( index, ListingModel::ModifyActionLabelRole ).toString() );
      } else {
        body += QString( "<td align=\"center\" %1><a href=\"%2\">%3</a></td>\n" )
                       .arg( colspan )
                       .arg( url.toString() )
                       .arg( mModel->data( index, ListingModel::ModifyActionLabelRole ).toString() );
      }
    }

    if ( mModel->data( index, ListingModel::HasDeleteActionRole ).toBool() ) {
      QUrl url;
      url.setScheme( "delete" );
      url.addQueryItem( "id", QString::number( row ) );

      const bool hasModifyAction = mModel->data( index, ListingModel::HasModifyActionRole ).toBool();
      const QString colspan = QString( "colspan=\"%1\"" ).arg( hasModifyAction ? "1" : "2" );

      /**
       * If the model returns an empty id, we don't show an active link.
       */
      if ( mModel->data( index, ListingModel::DeleteActionIdRole ).toString().isEmpty() ) {
        body += QString( "<td align=\"center\" %1>%2</td>\n" )
                       .arg( colspan )
                       .arg( mModel->data( index, ListingModel::DeleteActionLabelRole ).toString() );
      } else {
        body += QString( "<td align=\"center\" %1><a href=\"%2\">%3</a></td>\n" )
                       .arg( colspan )
                       .arg( url.toString() )
                       .arg( mModel->data( index, ListingModel::DeleteActionLabelRole ).toString() );
      }
    }

    body += "</tr>\n";
  }

  body += "</table>\n";
  body += "</center>\n";
  body += "</body>\n";
  body += "</html>\n";

  setHtml( body );
}

void ListingWidget::setSource( const QUrl &name )
{
  if ( name.scheme() == "modify" || name.scheme() == "delete" ) {
    int id = name.queryItemValue( "id" ).toInt();

    const QModelIndex index = mModel->index( id, 0 );
    if ( name.scheme() == "modify" )
      emit modifyEntry( mModel->data( index, ListingModel::ModifyActionIdRole ).toString() );
    else
      emit deleteEntry( mModel->data( index, ListingModel::DeleteActionIdRole ).toString() );
  } else {
      emit addEntry();
  }
}
