package theme

import (
	"github.com/darkhz/tview"
)

// ThemeProperty describes a theme property.
type ThemeProperty struct {
	Version int

	Context ThemeContext
	Item    ThemeItem

	IsForm bool
}

// ThemeContext describes the type of context to apply the color into.
type ThemeContext string

// The different context types for themes.
const (
	ThemeContextApp       ThemeContext = "App"
	ThemeContextMenu      ThemeContext = "Menu"
	ThemeContextStatusBar ThemeContext = "StatusBar"
	ThemeContextInstances ThemeContext = "Instances"
	ThemeContextLinks     ThemeContext = "Links"

	ThemeContextPlayerInfo   ThemeContext = "PlayerInfo"
	ThemeContextPlayer       ThemeContext = "Player"
	ThemeContextSearch       ThemeContext = "Search"
	ThemeContextDashboard    ThemeContext = "Dashboard"
	ThemeContextFiles        ThemeContext = "Files"
	ThemeContextDownloads    ThemeContext = "Downloads"
	ThemeContextQueue        ThemeContext = "Queue"
	ThemeContextFetcher      ThemeContext = "Fetcher"
	ThemeContextComments     ThemeContext = "Comments"
	ThemeContextStart        ThemeContext = "Start"
	ThemeContextPlaylist     ThemeContext = "Playlist"
	ThemeContextChannel      ThemeContext = "Channel"
	ThemeContextHistory      ThemeContext = "History"
	ThemeContextCustomSeeker ThemeContext = "CustomSeeker"
)

// ThemeItem describes a theme item.
type ThemeItem string

// The different item types for themes.
const (
	ThemeText            ThemeItem = "Text"
	ThemeBorder          ThemeItem = "Border"
	ThemeTitle           ThemeItem = "Title"
	ThemeTabs            ThemeItem = "Tabs"
	ThemeBackground      ThemeItem = "Background"
	ThemePopupBackground ThemeItem = "PopupBackground"
	ThemePopupBorder     ThemeItem = "PopupBorder"

	ThemeInputLabel ThemeItem = "InputLabel"
	ThemeInputField ThemeItem = "InputField"

	ThemeListLabel   ThemeItem = "ListLabel"
	ThemeListField   ThemeItem = "ListField"
	ThemeListOptions ThemeItem = "ListOptions"

	ThemeFormButton  ThemeItem = "FormButton"
	ThemeFormLabel   ThemeItem = "FormLabel"
	ThemeFormField   ThemeItem = "FormField"
	ThemeFormOptions ThemeItem = "FormOptions"

	ThemeSelector           ThemeItem = "Selector"
	ThemeNormalModeSelector ThemeItem = "NormalModeSelector"
	ThemeMoveModeSelector   ThemeItem = "MoveModeSelector"

	ThemeInfoMessage  ThemeItem = "InfoMessage"
	ThemeErrorMessage ThemeItem = "ErrorMessage"

	ThemeProgressBar  ThemeItem = "ProgressBar"
	ThemeProgressText ThemeItem = "ProgressText"

	ThemeTagStatusBar ThemeItem = "TagStatusBar"
	ThemeTagFetching  ThemeItem = "TagFetching"
	ThemeTagLoading   ThemeItem = "TagLoading"
	ThemeTagAdding    ThemeItem = "TagAdding"
	ThemeTagStopped   ThemeItem = "TagStopped"
	ThemeTagError     ThemeItem = "TagError"
	ThemeTagPlaying   ThemeItem = "TagPlaying"
	ThemeTagChanged   ThemeItem = "TagChanged"

	ThemeName        ThemeItem = "Name"
	ThemeDescription ThemeItem = "Description"
	ThemeKeybinding  ThemeItem = "Keybinding"

	ThemeDirectory ThemeItem = "Directory"
	ThemeFile      ThemeItem = "File"
	ThemePath      ThemeItem = "Path"

	ThemeVideo          ThemeItem = "Video"
	ThemePlaylist       ThemeItem = "Playlist"
	ThemeAuthor         ThemeItem = "Author"
	ThemeAuthorOwner    ThemeItem = "AuthorOwner"
	ThemeAuthorVerified ThemeItem = "AuthorVerified"
	ThemeTotalVideos    ThemeItem = "TotalVideos"

	ThemeShuffle       ThemeItem = "Shuffle"
	ThemeLoop          ThemeItem = "Loop"
	ThemeVolume        ThemeItem = "Volume"
	ThemeDuration      ThemeItem = "Duration"
	ThemeTotalDuration ThemeItem = "TotalDuration"
	ThemePause         ThemeItem = "Pause"
	ThemePlay          ThemeItem = "Play"
	ThemeBuffer        ThemeItem = "Buffer"
	ThemeMute          ThemeItem = "Mute"
	ThemeStop          ThemeItem = "Stop"

	ThemeChannel     ThemeItem = "Channel"
	ThemeComment     ThemeItem = "Comment"
	ThemeViews       ThemeItem = "Views"
	ThemeLikes       ThemeItem = "Likes"
	ThemeSubscribers ThemeItem = "Subscribers"
	ThemePublished   ThemeItem = "Published"

	ThemeInstanceURI  ThemeItem = "InstanceURI"
	ThemeInvidiousURI ThemeItem = "InvidiousURI"
	ThemeYoutubeURI   ThemeItem = "YoutubeURI"

	ThemeMediaInfo       ThemeItem = "MediaInfo"
	ThemeMediaSize       ThemeItem = "MediaSize"
	ThemeMediaType       ThemeItem = "MediaType"
	ThemeVideoResolution ThemeItem = "VideoResolution"
	ThemeVideoFPS        ThemeItem = "VideoFPS"
	ThemeAudioSampleRate ThemeItem = "AudioSampleRate"
	ThemeAudioChannels   ThemeItem = "AudioChannels"
)

// ThemeScopes store the ThemeItem scopes for each ThemeContext.
var ThemeScopes = map[ThemeContext]map[ThemeItem]struct{}{
	ThemeContextApp: {
		ThemeAuthor:          struct{}{},
		ThemeBackground:      struct{}{},
		ThemeBorder:          struct{}{},
		ThemeChannel:         struct{}{},
		ThemeDescription:     struct{}{},
		ThemeDuration:        struct{}{},
		ThemeErrorMessage:    struct{}{},
		ThemeFormButton:      struct{}{},
		ThemeFormField:       struct{}{},
		ThemeFormLabel:       struct{}{},
		ThemeFormOptions:     struct{}{},
		ThemeInfoMessage:     struct{}{},
		ThemeInputField:      struct{}{},
		ThemeInputLabel:      struct{}{},
		ThemeInstanceURI:     struct{}{},
		ThemeLikes:           struct{}{},
		ThemeListField:       struct{}{},
		ThemeListLabel:       struct{}{},
		ThemeListOptions:     struct{}{},
		ThemeMediaType:       struct{}{},
		ThemePlaylist:        struct{}{},
		ThemePopupBackground: struct{}{},
		ThemePopupBorder:     struct{}{},
		ThemeProgressBar:     struct{}{},
		ThemeProgressText:    struct{}{},
		ThemePublished:       struct{}{},
		ThemeSelector:        struct{}{},
		ThemeSubscribers:     struct{}{},
		ThemeTabs:            struct{}{},
		ThemeTagStatusBar:    struct{}{},
		ThemeText:            struct{}{},
		ThemeTitle:           struct{}{},
		ThemeTotalDuration:   struct{}{},
		ThemeTotalVideos:     struct{}{},
		ThemeVideo:           struct{}{},
		ThemeViews:           struct{}{},
	},
	ThemeContextChannel: {
		ThemeBackground:    struct{}{},
		ThemeBorder:        struct{}{},
		ThemeDescription:   struct{}{},
		ThemeMediaType:     struct{}{},
		ThemePlaylist:      struct{}{},
		ThemeSelector:      struct{}{},
		ThemeTabs:          struct{}{},
		ThemeTitle:         struct{}{},
		ThemeTotalDuration: struct{}{},
		ThemeTotalVideos:   struct{}{},
		ThemeVideo:         struct{}{},
	},
	ThemeContextComments: {
		ThemeAuthor:          struct{}{},
		ThemeAuthorOwner:     struct{}{},
		ThemeAuthorVerified:  struct{}{},
		ThemeComment:         struct{}{},
		ThemeLikes:           struct{}{},
		ThemePublished:       struct{}{},
		ThemePopupBorder:     struct{}{},
		ThemePopupBackground: struct{}{},
		ThemeSelector:        struct{}{},
		ThemeText:            struct{}{},
		ThemeTitle:           struct{}{},
		ThemeVideo:           struct{}{},
	},
	ThemeContextDashboard: {
		ThemeBorder:          struct{}{},
		ThemeBackground:      struct{}{},
		ThemeChannel:         struct{}{},
		ThemeFormButton:      struct{}{},
		ThemeFormField:       struct{}{},
		ThemeFormLabel:       struct{}{},
		ThemeFormOptions:     struct{}{},
		ThemeInputField:      struct{}{},
		ThemeInputLabel:      struct{}{},
		ThemeInstanceURI:     struct{}{},
		ThemeListField:       struct{}{},
		ThemeListLabel:       struct{}{},
		ThemeListOptions:     struct{}{},
		ThemePlaylist:        struct{}{},
		ThemePopupBorder:     struct{}{},
		ThemePopupBackground: struct{}{},
		ThemeSelector:        struct{}{},
		ThemeTabs:            struct{}{},
		ThemeText:            struct{}{},
		ThemeTotalDuration:   struct{}{},
		ThemeTotalVideos:     struct{}{},
		ThemeVideo:           struct{}{},
	},
	ThemeContextDownloads: {
		ThemeAudioChannels:   struct{}{},
		ThemeAudioSampleRate: struct{}{},
		ThemeBackground:      struct{}{},
		ThemeMediaInfo:       struct{}{},
		ThemeMediaSize:       struct{}{},
		ThemeMediaType:       struct{}{},
		ThemePopupBorder:     struct{}{},
		ThemePopupBackground: struct{}{},
		ThemeProgressBar:     struct{}{},
		ThemeProgressText:    struct{}{},
		ThemeSelector:        struct{}{},
		ThemeTitle:           struct{}{},
		ThemeVideoFPS:        struct{}{},
		ThemeVideoResolution: struct{}{},
	},
	ThemeContextFetcher: {
		ThemeAuthor:          struct{}{},
		ThemeErrorMessage:    struct{}{},
		ThemeInfoMessage:     struct{}{},
		ThemeMediaType:       struct{}{},
		ThemePopupBorder:     struct{}{},
		ThemePopupBackground: struct{}{},
		ThemeProgressText:    struct{}{},
		ThemeSelector:        struct{}{},
		ThemeTagAdding:       struct{}{},
		ThemeTagError:        struct{}{},
		ThemeTagStatusBar:    struct{}{},
		ThemeTitle:           struct{}{},
		ThemeVideo:           struct{}{},
	},
	ThemeContextFiles: {
		ThemeDirectory:       struct{}{},
		ThemeFile:            struct{}{},
		ThemeInputField:      struct{}{},
		ThemeInputLabel:      struct{}{},
		ThemePath:            struct{}{},
		ThemePopupBorder:     struct{}{},
		ThemePopupBackground: struct{}{},
		ThemeSelector:        struct{}{},
		ThemeTitle:           struct{}{},
	},
	ThemeContextHistory: {
		ThemeInputField:      struct{}{},
		ThemeInputLabel:      struct{}{},
		ThemeMediaType:       struct{}{},
		ThemePopupBorder:     struct{}{},
		ThemePopupBackground: struct{}{},
		ThemeSelector:        struct{}{},
		ThemeTitle:           struct{}{},
		ThemeVideo:           struct{}{},
	},
	ThemeContextInstances: {
		ThemeBackground:      struct{}{},
		ThemeInstanceURI:     struct{}{},
		ThemePopupBorder:     struct{}{},
		ThemePopupBackground: struct{}{},
		ThemeSelector:        struct{}{},
		ThemeTagChanged:      struct{}{},
		ThemeTitle:           struct{}{},
	},
	ThemeContextLinks: {
		ThemeInvidiousURI:    struct{}{},
		ThemePopupBorder:     struct{}{},
		ThemePopupBackground: struct{}{},
		ThemeSelector:        struct{}{},
		ThemeText:            struct{}{},
		ThemeTitle:           struct{}{},
		ThemeYoutubeURI:      struct{}{},
	},
	ThemeContextCustomSeeker: {
		ThemeFormField:       struct{}{},
		ThemeFormLabel:       struct{}{},
		ThemePopupBorder:     struct{}{},
		ThemePopupBackground: struct{}{},
		ThemeText:            struct{}{},
		ThemeTitle:           struct{}{},
	},
	ThemeContextMenu: {
		ThemeBackground:      struct{}{},
		ThemeBorder:          struct{}{},
		ThemeDescription:     struct{}{},
		ThemeKeybinding:      struct{}{},
		ThemeName:            struct{}{},
		ThemePopupBorder:     struct{}{},
		ThemePopupBackground: struct{}{},
		ThemeSelector:        struct{}{},
	},
	ThemeContextPlayer: {
		ThemeBackground:    struct{}{},
		ThemeBuffer:        struct{}{},
		ThemeDuration:      struct{}{},
		ThemeLoop:          struct{}{},
		ThemeMediaType:     struct{}{},
		ThemePause:         struct{}{},
		ThemePlay:          struct{}{},
		ThemeProgressBar:   struct{}{},
		ThemeSelector:      struct{}{},
		ThemeShuffle:       struct{}{},
		ThemeTitle:         struct{}{},
		ThemeTotalDuration: struct{}{},
		ThemeVolume:        struct{}{},
	},
	ThemeContextPlayerInfo: {
		ThemeAuthor:      struct{}{},
		ThemeBackground:  struct{}{},
		ThemeBorder:      struct{}{},
		ThemeDescription: struct{}{},
		ThemeLikes:       struct{}{},
		ThemeListField:   struct{}{},
		ThemeListLabel:   struct{}{},
		ThemeListOptions: struct{}{},
		ThemePublished:   struct{}{},
		ThemeSubscribers: struct{}{},
		ThemeTitle:       struct{}{},
		ThemeViews:       struct{}{},
	},
	ThemeContextPlaylist: {
		ThemeAuthor:        struct{}{},
		ThemeBackground:    struct{}{},
		ThemeBorder:        struct{}{},
		ThemeSelector:      struct{}{},
		ThemeTabs:          struct{}{},
		ThemeTotalDuration: struct{}{},
		ThemeTotalVideos:   struct{}{},
		ThemeVideo:         struct{}{},
	},
	ThemeContextQueue: {
		ThemeAuthor:             struct{}{},
		ThemeMediaType:          struct{}{},
		ThemeMoveModeSelector:   struct{}{},
		ThemeNormalModeSelector: struct{}{},
		ThemePopupBackground:    struct{}{},
		ThemePopupBorder:        struct{}{},
		ThemeSelector:           struct{}{},
		ThemeTagFetching:        struct{}{},
		ThemeTagLoading:         struct{}{},
		ThemeTagPlaying:         struct{}{},
		ThemeTagStopped:         struct{}{},
		ThemeTabs:               struct{}{},
		ThemeTitle:              struct{}{},
		ThemeTotalDuration:      struct{}{},
		ThemeVideo:              struct{}{},
	},
	ThemeContextSearch: {
		ThemeAuthor:          struct{}{},
		ThemeBackground:      struct{}{},
		ThemeBorder:          struct{}{},
		ThemeChannel:         struct{}{},
		ThemeFormButton:      struct{}{},
		ThemeFormField:       struct{}{},
		ThemeFormLabel:       struct{}{},
		ThemeFormOptions:     struct{}{},
		ThemeInputField:      struct{}{},
		ThemeInputLabel:      struct{}{},
		ThemePopupBackground: struct{}{},
		ThemePopupBorder:     struct{}{},
		ThemePlaylist:        struct{}{},
		ThemeSelector:        struct{}{},
		ThemeTabs:            struct{}{},
		ThemeTitle:           struct{}{},
		ThemeText:            struct{}{},
		ThemeTotalDuration:   struct{}{},
		ThemeTotalVideos:     struct{}{},
		ThemeVideo:           struct{}{},
	},
	ThemeContextStart: {
		ThemeText:       struct{}{},
		ThemeBackground: struct{}{},
	},
	ThemeContextStatusBar: {
		ThemeBackground:   struct{}{},
		ThemeErrorMessage: struct{}{},
		ThemeInfoMessage:  struct{}{},
		ThemeInputField:   struct{}{},
		ThemeInputLabel:   struct{}{},
		ThemeTagStatusBar: struct{}{},
	},
}

// SetItem sets the ThemeItem for the ThemeProperty.
func (t ThemeProperty) SetItem(item ThemeItem) ThemeProperty {
	t.Item = item

	return t
}

// SetContext sets the ThemeContext for the ThemeProperty.
func (t ThemeProperty) SetContext(context ThemeContext) ThemeProperty {
	t.Context = context

	return t
}

// UpdateThemeVersion updates the theme version.
func UpdateThemeVersion() {
	pconfig.Version++
}

// SetThemeProperty updates the ThemeProperty's version and applies the theme for its primitive.
func SetThemeProperty(primitive tview.Primitive, property *ThemeProperty, labelWidth ...int) {
	if property.Version == pconfig.Version {
		return
	}

	property.Version += 1

	applyTheme(primitive, *property, labelWidth...)
}
