/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iparticlesviewsubject.h"


#include "iactor.h"
#include "ibuffer.h"
#include "ierror.h"
#include "ifunctionmapping.h"
#include "imath.h"
#include "iparticleconnector.h"
#include "iparticlegroup.h"
#include "iparticlesplitter.h"

#include <vtkPolyData.h>

//
//  Templates
//
#include "iarraytemplate.h"
#include "ibuffertemplate.h"
#include "iviewfamilytemplate.h"


const int iParticlesViewSubject::CurrentId = iMath::_LargeInt;
bool iParticlesViewSubject::mUseFullState = true;


using namespace iParameter;


class iParticleGroupFamily : public iViewFamily<iParticleGroup,iParticlesViewSubject>
{
	
	friend class iParticlesViewSubject;

protected:
		
	iParticleGroupFamily(iParticlesViewSubject *p) : iViewFamily<iParticleGroup,iParticlesViewSubject>(p)
	{
		mParent = p;
		mMemlist[0]->mParentIndex = 0;
	}

	virtual int CreateMember()
	{
		int ret = iViewFamily<iParticleGroup,iParticlesViewSubject>::CreateMember();
		if(ret >= 0)
		{
			mMemlist[ret]->mParentIndex = ret;
		}
		return ret;
	}

	virtual bool DeleteMember(int i)
	{
		bool ret = iViewFamily<iParticleGroup,iParticlesViewSubject>::DeleteMember(i);
		if(ret)
		{
			int j;
			for(j=i; j<mMemlist.Size(); j++) mMemlist[j]->mParentIndex = j;
		}
		return ret;
	}

private:

	iParticlesViewSubject *mParent;
};


//
//  Main class
//
IVIEWSUBJECT_DEFINE_TYPE(iParticlesViewSubject,Particles,p);

IOBJECT_DEFINE_KEY(iParticlesViewSubject,AttributeToSplit,a,OffsetInt,1);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,AttributeToColor,ac,OffsetInt,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,AttributeSizeDirect,ad,Bool,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,AttributeSizeExtraFactor,ae,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,AttributeToSize,as,OffsetInt,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,AttributeToConnect,atc,OffsetInt,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,AttributeToSeparate,ats,OffsetInt,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,AutoScaled,au,Bool,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,Color,c,Color,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,CurrentGroup,cg,OffsetInt,1);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,StretchToColor,stc,Int,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,LowerLimitToColor,ltc,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,UpperLimitToColor,utc,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,StretchToSize,sts,Int,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,LowerLimitToSize,lts,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,UpperLimitToSize,uts,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,LineWidth,lw,Int,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,MaxGroup,mg,OffsetInt,1);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,Opacity,o,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,Palette,p,Int,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,SplitRangeLowerLimit,rl,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,SplitRangesMax,rmax,Float,1);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,SplitRangesMin,rmin,Float,1);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,SplitRangesStretch,rs,Int,1);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,SplitRangesTiled,rt,Bool,1);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,SplitRangeUpperLimit,ru,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,FixedSize,s,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,SizeFunction,sf,Any,0);
IOBJECT_DEFINE_KEY(iParticlesViewSubject,Type,t,Int,0);

IOBJECT_DEFINE_KEY(iParticlesViewSubject,SplitRanges,-r,Any,1);

//
//  Inherited keys (define them explicitly since they are now multi-component
//
IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,iParticlesViewSubject,ClipPlane,cp,Bool,1);
IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,iParticlesViewSubject,Ambient,ma,Float,0);
IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,iParticlesViewSubject,Diffuse,md,Float,0);
IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,iParticlesViewSubject,Specular,ms,Float,0);
IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,iParticlesViewSubject,SpecularPower,mp,Float,0);
IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,iParticlesViewSubject,Visible,vis,Bool,1);
IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,iParticlesViewSubject,DataPresent,dp,Bool,1);
IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,iParticlesViewSubject,Ready,r,Bool,1);
IOBJECT_DEFINE_INHERITED_KEY(iViewSubject,iParticlesViewSubject,Shading,mw,Bool,0);

IVIEWSUBJECT_DEFINE_INHERITED_KEYS_REPLICATING(iParticlesViewSubject);


iParticlesViewSubject::iParticlesViewSubject(iViewModule *vm, const iDataType &type, const iString &name, const iColor &color, bool shaded) : iViewSubject(vm,type,name,0U)
{
	mSubjectId = ViewSubject::Id::Particles;

	mGroups = new iParticleGroupFamily(this); IERROR_ASSERT(mGroups);
	mSplitter = iParticleSplitter::New(this); IERROR_ASSERT(mSplitter);

	if(color.IsValid()) mGroups->GetMember(0)->SetColor(0,color);
	mGroups->GetMember(0)->SetShading(shaded);
}


iParticlesViewSubject::~iParticlesViewSubject()
{
	delete mGroups;
	mSplitter->Delete();
}


void iParticlesViewSubject::ConfigureBody()
{
	//
	//  Attach the first group
	//
	mGroups->GetMember(0)->SetInput(mSplitter->GetOutput());
}


void iParticlesViewSubject::BecomeClone(iParticlesViewSubject *v)
{
	mSplitter->TakeOverData(v != 0);
	this->ClearCache();
}


void iParticlesViewSubject::UpdateReplicas()
{
	int i;
	for(i=0; i<=mGroups->GetMaxMemberIndex(); i++) 
	{
		mGroups->GetMember(i)->ReplicateAs(this);
	}
}


void iParticlesViewSubject::ShowBody(bool s)
{
	mGroups->Show(s);
}


void iParticlesViewSubject::Reset()
{
	mGroups->Reset();
	this->SyncWithData(this->RequestAll());
	this->ClearCache();
}


void iParticlesViewSubject::ShowClipPlane(bool s)
{
	int i;
	mClipPlaneOn = s;
	for(i=0; i<=mGroups->GetMaxMemberIndex(); i++) mGroups->GetMember(i)->ShowClipPlane(s);
	this->ClearCache();
}


void iParticlesViewSubject::ShowColorBars(bool s)
{
	int i;
	for(i=0; i<=mGroups->GetMaxMemberIndex(); i++) mGroups->GetMember(i)->ShowColorBars(s);
	this->ClearCache();
}


void iParticlesViewSubject::PackStateBody(iString &s) const
{
	static iBuffer<float> fv1, fv2;
	int i, n = mGroups->GetMaxMemberIndex() + 1;
	float f1, f2;

	this->PackValue(s,KeyMaxGroup(),mGroups->GetMaxMemberIndex()); // arrays start with 1
	this->PackValue(s,KeyCurrentGroup(),mGroups->GetCurrentMemberIndex()); // arrays start with 1

	this->PackValue(s,KeyClipPlane(),mClipPlaneOn);
	this->PackValue(s,KeyNumReplicas(),mNumReplicas,6);
	this->PackValue(s,KeyVisible(),this->IsVisible());
	this->PackValue(s,KeyDataPresent(),this->IsThereData());
	this->PackValue(s,KeyReady(),this->IsReady());

	this->PackValue(s,KeyAttributeToSplit(),mSplitter->GetAttributeToSplit());
	this->PackValue(s,KeySplitRangesStretch(),mSplitter->GetSplitRangesStretch());
	this->PackValue(s,KeySplitRangesTiled(),mSplitter->GetSplitRangesTiled());

	mSplitter->GetSplitRangesMinMax(f1,f2);
	this->PackValue(s,KeySplitRangesMin(),f1);
	this->PackValue(s,KeySplitRangesMax(),f2);

	for(i=0; i<n; i++)
	{
		mSplitter->GetSplitRangeLimits(i,fv1[i],fv2[i]);
	}
	this->PackValue(s,KeySplitRangeLowerLimit(),fv1,n);
	this->PackValue(s,KeySplitRangeUpperLimit(),fv2,n);

	if(mUseFullState) this->PackGroupStateBody(s,0,mGroups->GetMaxMemberIndex());
}


void iParticlesViewSubject::UnPackStateBody(const iString &s0)
{
	static iString rep1 = iObjectKey::LeftBracket() + iObjectKey::RightBracket();
	static iString rep2 = iObjectKey::LeftBracket() + "0" + iObjectKey::RightBracket();
	static iString rep3 = iObjectKey::LeftBracket() + iString::FromNumber(1+CurrentId) + iObjectKey::RightBracket();
	static iBuffer<float> fv1, fv2;
	int i, n, v, d[6];
	bool b;

	//
	//  Unpack splitter attribute first, so that groups are created correctly
	//
	if(this->UnPackValue(s0,KeyAttributeToSplit(),i)) this->SetAttributeToSplit(i);
	//
	//  Make the size of the group family consistent
	//
	if(this->UnPackValue(s0,KeyMaxGroup(),v))
	{
		//
		//  Delete extra groups
		//
		while(v < mGroups->GetMaxMemberIndex()) this->DeleteGroup(mGroups->GetMaxMemberIndex());
		//
		//  Create missing groups
		//
		while(v > mGroups->GetMaxMemberIndex()) this->CreateGroup();
		this->ClearCache();
	}

	n = mGroups->GetMaxMemberIndex() + 1;

	if(this->UnPackValue(s0,KeyCurrentGroup(),v))
	{
		mGroups->SetCurrentMemberIndex(v); // arrays start with 1
		this->ClearCache();
	}

	//
	//  Special syntax: replace [] or [0] with [<curindex>].
	//
	iString sub, s = s0;
	sub = iObjectKey::LeftBracket() + iString::FromNumber(1+mGroups->GetCurrentMemberIndex()) + iObjectKey::RightBracket(); // arrays start with 1

	s.Replace(rep1,sub);
	s.Replace(rep2,sub);
	s.Replace(rep3,sub);

	if(this->UnPackValue(s,KeyClipPlane(),b)) this->ShowClipPlane(b);

	for(i=0; i<6; i++) d[i] = mNumReplicas[i];
	if(this->UnPackValue(s,KeyNumReplicas(),d,6)) this->SetNumReplicas(d);

	if(this->UnPackValue(s,KeySplitRangesStretch(),i))
	{
		mSplitter->SetSplitRangesStretch(i);
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeySplitRangesTiled(),b))
	{
		mSplitter->SetSplitRangesTiled(b);
		this->ClearCache();
	}

	float f1, f2;
	if(this->UnPackValue(s,KeySplitRangesMin(),f1) && this->UnPackValue(s,KeySplitRangesMax(),f2))
	{
		mSplitter->SetSplitRangesMinMax(f1,f2);
		this->ClearCache();
	}

	for(i=0; i<n; i++)
	{
		mSplitter->GetSplitRangeLimits(i,fv1[i],fv2[i]);
	}
	if(this->UnPackValue(s,KeySplitRangeLowerLimit(),fv1,n) && this->UnPackValue(s,KeySplitRangeUpperLimit(),fv2,n)) for(i=0; i<n; i++)
	{
		mSplitter->SetSplitRangeLimits(i,fv1[i],fv2[i]);
		this->ClearCache();
	}
	
	if(mUseFullState) this->UnPackGroupStateBody(s,0,mGroups->GetMaxMemberIndex());
}


void iParticlesViewSubject::PackCompleteState(iString &s) const
{
	bool u = mUseFullState;

	mUseFullState = true;
	mCacheInvalid = true;
	this->PackState(s);
	mUseFullState = u;
}


void iParticlesViewSubject::UnPackCompleteState(const iString &s)
{
	bool u = mUseFullState;

	mUseFullState = true;
	this->UnPackState(s);
	mUseFullState = u;
}


void iParticlesViewSubject::PackGroupStateBody(iString &s, int first, int last) const
{
	static iBuffer<int> iv;
	static iBuffer<bool> bv;
	static iBuffer<float> fv;
	static iBuffer<double> dv;
	static iBuffer<iColor> cv;
	static iBuffer<const iPiecewiseFunction*> pv;
	int i, n;
	
	if(last > mGroups->GetMaxMemberIndex()) last = mGroups->GetMaxMemberIndex();
	n = last - first + 1;
	if(n < 1) return;

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetType();
	this->PackValue(s,KeyType(),iv+first,n);

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetPaletteId(0);
	this->PackValue(s,KeyPalette(),iv+first,n);

	for(i=first; i<=last; i++) cv[i] = mGroups->GetMember(i)->GetColor(0);
	this->PackValue(s,KeyColor(),cv+first,n);

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetAttributeToColor();
	this->PackValue(s,KeyAttributeToColor(),iv+first,n);

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetAttributeToSize();
	this->PackValue(s,KeyAttributeToSize(),iv+first,n);

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetOpacity(0);
	this->PackValue(s,KeyOpacity(),fv+first,n);

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetFixedSize();
	this->PackValue(s,KeyFixedSize(),fv+first,n);

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetStretchToColor();
	this->PackValue(s,KeyStretchToColor(),iv+first,n);

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetLowerLimitToColor();
	this->PackValue(s,KeyLowerLimitToColor(),fv+first,n);

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetUpperLimitToColor();
	this->PackValue(s,KeyUpperLimitToColor(),fv+first,n);

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetStretchToSize();
	this->PackValue(s,KeyStretchToSize(),iv+first,n);

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetLowerLimitToSize();
	this->PackValue(s,KeyLowerLimitToSize(),fv+first,n);

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetUpperLimitToSize();
	this->PackValue(s,KeyUpperLimitToSize(),fv+first,n);

	for(i=first; i<=last; i++) bv[i] = mGroups->GetMember(i)->GetAttributeSizeDirect();
	this->PackValue(s,KeyAttributeSizeDirect(),bv+first,n);

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetAttributeSizeExtraFactor();
	this->PackValue(s,KeyAttributeSizeExtraFactor(),fv+first,n);

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetLineWidth();
	this->PackValue(s,KeyLineWidth(),iv+first,n);

	for(i=first; i<=last; i++) bv[i] = mGroups->GetMember(i)->GetAutoScaled();
	this->PackValue(s,KeyAutoScaled(),bv+first,n);

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetAttributeToConnect();
	this->PackValue(s,KeyAttributeToConnect(),iv+first,n);

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetAttributeToSeparate();
	this->PackValue(s,KeyAttributeToSeparate(),iv+first,n);

	for(i=first; i<=last; i++) pv[i] = mGroups->GetMember(i)->GetSizeFunction()->GetFunction();
	this->PackValuePiecewiseFunction(s,KeySizeFunction(),pv+first,n);

	for(i=first; i<=last; i++) bv[i] = mGroups->GetMember(i)->GetShading();
	this->PackValue(s,KeyShading(),bv+first,n);

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetAmbient();
	this->PackValue(s,KeyAmbient(),fv+first,n);
	
	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetDiffuse();
	this->PackValue(s,KeyDiffuse(),fv+first,n);
	
	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetSpecular();
	this->PackValue(s,KeySpecular(),fv+first,n);
	
	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetSpecularPower();
	this->PackValue(s,KeySpecularPower(),fv+first,n);
}


void iParticlesViewSubject::UnPackGroupStateBody(const iString &s, int first, int last)
{
	static iBuffer<int> iv;
	static iBuffer<bool> bv;
	static iBuffer<float> fv;
	static iBuffer<double> dv;
	static iBuffer<iColor> cv;
	static iBuffer<iPiecewiseFunction*> pv;
	int i, n;

	if(last > mGroups->GetMaxMemberIndex()) last = mGroups->GetMaxMemberIndex();
	n = last - first + 1;
	if(n < 1) return;

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetType();
	if(this->UnPackValue(s,KeyType(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetType(iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetPaletteId(0);
	if(this->UnPackValue(s,KeyPalette(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetPaletteId(0,iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) cv[i] = mGroups->GetMember(i)->GetColor(0);
	if(this->UnPackValue(s,KeyColor(),cv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetColor(0,cv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetAttributeToColor();
	if(this->UnPackValue(s,KeyAttributeToColor(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetAttributeToColor(iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetAttributeToSize();
	if(this->UnPackValue(s,KeyAttributeToSize(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetAttributeToSize(iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetOpacity(0);
	if(this->UnPackValue(s,KeyOpacity(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetOpacity(0,fv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetFixedSize();
	if(this->UnPackValue(s,KeyFixedSize(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetFixedSize(fv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetStretchToColor();
	if(this->UnPackValue(s,KeyStretchToColor(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetStretchToColor(iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetLowerLimitToColor();
	if(this->UnPackValue(s,KeyLowerLimitToColor(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetLowerLimitToColor(fv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetUpperLimitToColor();
	if(this->UnPackValue(s,KeyUpperLimitToColor(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetUpperLimitToColor(fv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetStretchToSize();
	if(this->UnPackValue(s,KeyStretchToSize(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetStretchToSize(iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetLowerLimitToSize();
	if(this->UnPackValue(s,KeyLowerLimitToSize(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetLowerLimitToSize(fv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetUpperLimitToSize();
	if(this->UnPackValue(s,KeyUpperLimitToSize(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetUpperLimitToSize(fv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) bv[i] = mGroups->GetMember(i)->GetAttributeSizeDirect();
	if(this->UnPackValue(s,KeyAttributeSizeDirect(),bv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetAttributeSizeDirect(bv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetAttributeSizeExtraFactor();
	if(this->UnPackValue(s,KeyAttributeSizeExtraFactor(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetAttributeSizeExtraFactor(fv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetLineWidth();
	if(this->UnPackValue(s,KeyLineWidth(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetLineWidth(iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) bv[i] = mGroups->GetMember(i)->GetAutoScaled();
	if(this->UnPackValue(s,KeyAutoScaled(),bv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetAutoScaled(bv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetAttributeToConnect();
	if(this->UnPackValue(s,KeyAttributeToConnect(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetAttributeToConnect(iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetAttributeToSeparate();
	if(this->UnPackValue(s,KeyAttributeToSeparate(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetAttributeToSeparate(iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) pv[i] = mGroups->GetMember(i)->GetSizeFunction()->GetFunction();
	if(this->UnPackValuePiecewiseFunction(s,KeySizeFunction(),pv+first,n)) this->ClearCache(); // this unpacks into the appropriate function automatically.

	for(i=first; i<=last; i++) bv[i] = mGroups->GetMember(i)->GetShading();
	if(this->UnPackValue(s,KeyShading(),bv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetShading(bv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetAmbient();
	if(this->UnPackValue(s,KeyAmbient(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetAmbient(fv[i]);
		this->ClearCache();
	}
	
	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetDiffuse();
	if(this->UnPackValue(s,KeyDiffuse(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetDiffuse(fv[i]);
		this->ClearCache();
	}
	
	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetSpecular();
	if(this->UnPackValue(s,KeySpecular(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetSpecular(fv[i]);
		this->ClearCache();
	}
	
	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetSpecularPower();
	if(this->UnPackValue(s,KeySpecularPower(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetSpecularPower(fv[i]);
		this->ClearCache();
	}
}


float iParticlesViewSubject::GetMemorySize() const
{
	return mGroups->GetMemorySize() + mSplitter->GetMemorySize();
}


bool iParticlesViewSubject::IsVisible() const
{
	return mGroups->IsVisible();
}


int iParticlesViewSubject::GetMaxGroupIndex() const
{
	return mGroups->GetMaxMemberIndex();
}


int iParticlesViewSubject::GetCurrentGroupIndex() const
{
	return mGroups->GetCurrentMemberIndex();
}


iParticleGroup* iParticlesViewSubject::GetCurrentGroup() const
{
	return mGroups->GetCurrentMember();
}


iParticleGroup* iParticlesViewSubject::GetGroup(int i) const
{
	return mGroups->GetMember(i);
}


iRangeMapping* iParticlesViewSubject::GetSplitRanges()
{
	return mSplitter->GetSplitRanges();
}


void iParticlesViewSubject::SetCurrentGroupIndex(int i)
{
	mGroups->SetCurrentMemberIndex(i);
}


int iParticlesViewSubject::CreateGroup()
{ 
	if(!mSplitter->CreateGroup()) return -1;

	vtkPolyData *newOutput = mSplitter->GetOutput(mSplitter->GetNumberOfOutputs()-1);
	//
	//  Create a new piece
	//
	int oldGroup = mGroups->GetCurrentMemberIndex();
	int newGroup = mGroups->CreateMember();
	if(newGroup == -1)
	{
		mSplitter->DeleteGroup(mSplitter->GetNumberOfOutputs()-1);
		return -1;
	}

	//
	//  Set the correct piece number
	//
	mGroups->GetMember(newGroup)->CopyState(mGroups->GetMember(oldGroup));
	mGroups->GetMember(newGroup)->SetInput(newOutput);

	mGroups->GetMember(newGroup)->Show(mGroups->GetMember(oldGroup)->IsVisible());
	mGroups->SetCurrentMemberIndex(newGroup);
	
	this->ClearCache();

	return newGroup;
}


bool iParticlesViewSubject::DeleteGroup(int n)
{
	if(mSplitter->DeleteGroup(n))
	{
		this->ClearCache();
		if(mGroups->DeleteMember(n))
		{
			//
			//  Need to update inputs
			//
			int i;
			for(i=n; i<=mGroups->GetMaxMemberIndex(); i++) mGroups->GetMember(i)->SetInput(mSplitter->GetOutput(i));
			return true;
		}
		else return false;
	}
	else return false;
}


void iParticlesViewSubject::UpdateColorLookupTable()
{
	int i;
	for(i=0; i<=mGroups->GetMaxMemberIndex(); i++) mGroups->GetMember(i)->UpdateColorLookupTable();
}


void iParticlesViewSubject::UpdateMaterialProperties()
{
	int i;
	for(i=0; i<=mGroups->GetMaxMemberIndex(); i++) mGroups->GetMember(i)->UpdateMaterialProperties();
}


void iParticlesViewSubject::SetAttributeToSplit(int a)
{
	int oldAtt = mSplitter->GetAttributeToSplit();
	mSplitter->SetAttributeToSplit(a);
	if(oldAtt != mSplitter->GetAttributeToSplit())
	{
		while(mGroups->GetMaxMemberIndex() > 0) this->DeleteGroup(mGroups->GetMaxMemberIndex());
		mGroups->GetMember(0)->UpdateLookupTables();
	}
	this->ClearCache();
}


int iParticlesViewSubject::GetAttributeToSplit() const
{
	return mSplitter->GetAttributeToSplit();
}


void iParticlesViewSubject::ViewSubjectPackStateBody(iString &) const
{
#ifdef I_CHECK1
	IERROR_LOW("This function should never be called.");
#endif
}


void iParticlesViewSubject::ViewSubjectUnPackStateBody(const iString &)
{
#ifdef I_CHECK1
	IERROR_LOW("This function should never be called.");
#endif
}


bool iParticlesViewSubject::CanBeShown() const
{
	return (vtkPolyData::SafeDownCast(this->GetData()) != 0);
}


void iParticlesViewSubject::ViewSubjectSyncWithData(const iDataSyncRequest &r)
{
	int i;

	mSplitter->SetInput(vtkPolyData::SafeDownCast(this->GetData()));

	for(i=0; i<=mGroups->GetMaxMemberIndex(); i++) 
	{
		mGroups->GetMember(i)->ViewSubjectSyncWithData(r);
	}
}

